# Repository Guidelines

## Project Structure & Module Organization
The service uses a standard NestJS layout rooted in `src/`. `src/main.ts` bootstraps the HTTP server via `AppModule`. Domain modules live in feature folders such as `product/`, `purchase-order/`, `supplier/`, and `supplier-product/`. Cross-cutting helpers stay under `src/common/`, while persistence logic is grouped in `src/entities/` and `src/database/`. TypeORM migration classes reside in `src/migrations/`; generated files should be checked in. Use `scripts/` for CLI helpers (e.g., migration generation). Keep tests close to their targets: unit specs beside the implementation (`*.spec.ts`), and e2e suites inside `test/`. The `dist/` directory is build output and should remain untracked locally.

## Build, Test, and Development Commands
Run `npm install` once to sync dependencies. Everyday commands:
- `npm run start:dev` – launch the NestJS server with reload.
- `npm run start:prod` – execute the compiled bundle from `dist/`.
- `npm run build` – transpile TypeScript to `dist/`.
- `npm run lint` / `npm run format` – enforce ESLint and Prettier rules.
- `npm run migration:generate -- src/migrations/<Name>` and `npm run migration:run` – manage TypeORM schema changes.
- `npm run seed` – populate sample data after migrations.

## Coding Style & Naming Conventions
All code is TypeScript with 2-space indentation. Rely on the repo’s `eslint.config.mjs` and Prettier defaults; run `npm run lint` before committing and `npm run format` for bulk formatting. Favor PascalCase class names (`PurchaseOrderService`), camelCase methods and variables, and UPPER_SNAKE_CASE for environment flags. DTO files end with `.dto.ts`, entities with `.entity.ts`, and providers with `.service.ts`. Keep module files exporting a single Nest module per folder.

## Testing Guidelines
Jest drives both unit and e2e tests (`jest-e2e.json`). Name unit specs `*.spec.ts` alongside their source and e2e suites `*.e2e-spec.ts` under `test/`. Use `npm run test` for the full unit suite, `npm run test:watch` while iterating, and `npm run test:e2e` when exercising HTTP flows. `npm run test:cov` writes coverage reports to `coverage/`; aim to keep new code at ≥80% line coverage and justify exceptions. Mock external services via Nest testing modules rather than hitting live integrations.

## Commit & Pull Request Guidelines
Adopt Conventional Commits (`feat: add supplier filters`, `fix: correct tax rounding`) to keep the history searchable. Commits should be scoped and lint-clean. Pull requests include a concise summary of changes, a deployment/test checklist, linked tickets, and notes on migrations or environment variables. Attach screenshots or sample responses for API-affecting work and ensure new endpoints are reflected in Swagger decorators.

## Database & Migration Workflow
Database settings resolve from `typeorm.config.ts` and `.env`. Before generating migrations, ensure the target database schema reflects current entities. Always review generated SQL for accuracy, commit the migration class, and rerun `npm run migration:run` plus any relevant seed scripts in CI-like conditions.
