"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PurchaseOrderService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const purchase_order_entity_1 = require("../entities/purchase-order.entity");
const typeorm_2 = require("typeorm");
const purchase_order_item_entity_1 = require("../entities/purchase-order-item.entity");
const supplier_entity_1 = require("../entities/supplier.entity");
const product_entity_1 = require("../entities/product.entity");
const stock_service_1 = require("../stock/stock.service");
const size_entity_1 = require("../entities/size.entity");
let PurchaseOrderService = class PurchaseOrderService {
    purchaseOrderRepository;
    purchaseOrderItemRepository;
    supplierRepository;
    dataSource;
    stockService;
    constructor(purchaseOrderRepository, purchaseOrderItemRepository, supplierRepository, dataSource, stockService) {
        this.purchaseOrderRepository = purchaseOrderRepository;
        this.purchaseOrderItemRepository = purchaseOrderItemRepository;
        this.supplierRepository = supplierRepository;
        this.dataSource = dataSource;
        this.stockService = stockService;
    }
    async createPurchaseOrder(createPurchaseOrderDto) {
        const queryRunner = this.dataSource.createQueryRunner();
        await queryRunner.connect();
        await queryRunner.startTransaction();
        try {
            const { supplier_id, order_date, items } = createPurchaseOrderDto;
            const supplier = await queryRunner.manager.findOneBy(supplier_entity_1.Supplier, {
                id: supplier_id,
            });
            if (!supplier) {
                throw new common_1.NotFoundException(`Supplier with ID ${supplier_id} not found`);
            }
            const purchaseOrder = queryRunner.manager.create(purchase_order_entity_1.PurchaseOrder, {
                supplier,
                order_date: new Date(order_date),
                total_amount: 0,
            });
            const savedOrder = await queryRunner.manager.save(purchaseOrder);
            let totalOrderAmount = 0;
            for (const itemDto of items) {
                const product = await queryRunner.manager.findOneBy(product_entity_1.Product, {
                    id: itemDto.product_id,
                });
                if (!product) {
                    throw new common_1.NotFoundException(`Product with ID ${itemDto.product_id} not found`);
                }
                const size = await queryRunner.manager.findOne(size_entity_1.Size, {
                    where: { id: itemDto.size_id },
                });
                if (!size) {
                    throw new common_1.NotFoundException(`Size with ID ${itemDto.size_id} not found`);
                }
                const itemTotal = itemDto.quantity * itemDto.unit_price;
                totalOrderAmount += itemTotal;
                const orderItem = queryRunner.manager.create(purchase_order_item_entity_1.PurchaseOrderItem, {
                    purchaseOrder: savedOrder,
                    product,
                    size,
                    size_id: size.id,
                    quantity: itemDto.quantity,
                    unit_price: itemDto.unit_price,
                    total: itemTotal,
                });
                await queryRunner.manager.save(orderItem);
                await this.stockService.increaseStock({
                    productId: product.id,
                    sizeId: size.id,
                    quantity: itemDto.quantity,
                    movementType: "NEW",
                    manager: queryRunner.manager,
                    referenceType: "PURCHASE_ORDER",
                    referenceId: savedOrder.id,
                });
            }
            savedOrder.total_amount = totalOrderAmount;
            await queryRunner.manager.save(savedOrder);
            await queryRunner.commitTransaction();
            return this.getPurchaseOrderById(savedOrder.id);
        }
        catch (err) {
            await queryRunner.rollbackTransaction();
            throw err;
        }
        finally {
            await queryRunner.release();
        }
    }
    async updatePurchaseOrder(id, updatePurchaseOrderDto) {
        const queryRunner = this.dataSource.createQueryRunner();
        await queryRunner.connect();
        await queryRunner.startTransaction();
        try {
            let purchaseOrder = await queryRunner.manager.findOne(purchase_order_entity_1.PurchaseOrder, {
                where: { id },
                relations: [
                    "supplier",
                    "purchaseOrderItems",
                    "purchaseOrderItems.product",
                    "purchaseOrderItems.size",
                ],
            });
            if (!purchaseOrder) {
                throw new common_1.NotFoundException(`Purchase Order with ID ${id} not found`);
            }
            const originalStockSnapshot = this.buildStockSnapshot(purchaseOrder.purchaseOrderItems);
            if (updatePurchaseOrderDto.supplier_id) {
                const supplier = await queryRunner.manager.findOneBy(supplier_entity_1.Supplier, {
                    id: updatePurchaseOrderDto.supplier_id,
                });
                if (!supplier) {
                    throw new common_1.NotFoundException(`Supplier with ID ${updatePurchaseOrderDto.supplier_id} not found`);
                }
                purchaseOrder.supplier = supplier;
            }
            if (updatePurchaseOrderDto.order_date) {
                purchaseOrder.order_date = new Date(updatePurchaseOrderDto.order_date);
            }
            if (updatePurchaseOrderDto.items) {
                let totalOrderAmount = 0;
                const existingItemIds = purchaseOrder.purchaseOrderItems.map((item) => item.id);
                const updatedItemIds = updatePurchaseOrderDto.items
                    .filter((item) => item.id)
                    .map((item) => item.id);
                const itemsToRemove = existingItemIds.filter((id) => !updatedItemIds.includes(id));
                if (itemsToRemove.length > 0) {
                    await queryRunner.manager.delete(purchase_order_item_entity_1.PurchaseOrderItem, itemsToRemove);
                }
                for (const itemDto of updatePurchaseOrderDto.items) {
                    let orderItem;
                    if (itemDto.id) {
                        orderItem = purchaseOrder.purchaseOrderItems.find((item) => item.id === itemDto.id);
                        if (!orderItem) {
                            throw new common_1.NotFoundException(`Purchase Order Item with ID ${itemDto.id} not found in order ${id}`);
                        }
                    }
                    else {
                        orderItem = queryRunner.manager.create(purchase_order_item_entity_1.PurchaseOrderItem, {
                            purchaseOrder,
                        });
                    }
                    if (itemDto.product_id) {
                        const product = await queryRunner.manager.findOneBy(product_entity_1.Product, {
                            id: itemDto.product_id,
                        });
                        if (!product) {
                            throw new common_1.NotFoundException(`Product with ID ${itemDto.product_id} not found`);
                        }
                        orderItem.product = product;
                    }
                    if (itemDto.size_id) {
                        const size = await queryRunner.manager.findOne(size_entity_1.Size, {
                            where: { id: itemDto.size_id },
                        });
                        if (!size) {
                            throw new common_1.NotFoundException(`Size with ID ${itemDto.size_id} not found`);
                        }
                        orderItem.size = size;
                        orderItem.size_id = size.id;
                    }
                    if (itemDto.quantity)
                        orderItem.quantity = itemDto.quantity;
                    if (itemDto.unit_price)
                        orderItem.unit_price = itemDto.unit_price;
                    orderItem.total = orderItem.quantity * orderItem.unit_price;
                    totalOrderAmount += orderItem.total;
                    await queryRunner.manager.save(orderItem);
                }
                purchaseOrder.total_amount = totalOrderAmount;
            }
            else {
                purchaseOrder = await queryRunner.manager.findOne(purchase_order_entity_1.PurchaseOrder, {
                    where: { id },
                    relations: ["purchaseOrderItems", "purchaseOrderItems.size"],
                });
                if (!purchaseOrder) {
                    throw new common_1.NotFoundException(`Purchase Order with ID ${id} not found`);
                }
                purchaseOrder.total_amount = purchaseOrder.purchaseOrderItems.reduce((sum, item) => sum + item.total, 0);
            }
            await queryRunner.manager.save(purchaseOrder);
            const updatedOrderState = await queryRunner.manager.findOne(purchase_order_entity_1.PurchaseOrder, {
                where: { id },
                relations: [
                    "purchaseOrderItems",
                    "purchaseOrderItems.product",
                    "purchaseOrderItems.size",
                ],
            });
            if (!updatedOrderState) {
                throw new common_1.NotFoundException(`Purchase Order with ID ${id} not found after update`);
            }
            await this.applyStockDiff(originalStockSnapshot, this.buildStockSnapshot(updatedOrderState.purchaseOrderItems), queryRunner.manager, id);
            await queryRunner.commitTransaction();
            return this.getPurchaseOrderById(id);
        }
        catch (err) {
            await queryRunner.rollbackTransaction();
            throw err;
        }
        finally {
            await queryRunner.release();
        }
    }
    async findAllPurchaseOrders() {
        return this.purchaseOrderRepository.find({
            relations: [
                "supplier",
                "purchaseOrderItems",
                "purchaseOrderItems.product",
                "purchaseOrderItems.size",
            ],
        });
    }
    async getPurchaseOrderById(id) {
        const order = await this.purchaseOrderRepository.findOne({
            where: { id },
            relations: [
                "supplier",
                "purchaseOrderItems",
                "purchaseOrderItems.product",
                "purchaseOrderItems.size",
            ],
        });
        if (!order) {
            throw new common_1.NotFoundException(`Purchase Order with ID ${id} not found`);
        }
        return order;
    }
    async findPurchaseOrdersBySupplier(supplier_id) {
        const supplier = await this.supplierRepository.findOneBy({
            id: supplier_id,
        });
        if (!supplier) {
            throw new common_1.NotFoundException(`Supplier with ID ${supplier_id} not found`);
        }
        return this.purchaseOrderRepository.find({
            where: { supplier: { id: supplier_id } },
            relations: [
                "supplier",
                "purchaseOrderItems",
                "purchaseOrderItems.product",
                "purchaseOrderItems.size",
            ],
        });
    }
    async deletePurchaseOrder(id) {
        const result = await this.purchaseOrderRepository.delete(id);
        if (result.affected === 0) {
            throw new common_1.NotFoundException(`Purchase Order with ID ${id} not found`);
        }
    }
    buildStockSnapshot(items) {
        const snapshot = new Map();
        for (const item of items) {
            if (!item.product || !item.size) {
                continue;
            }
            const key = `${item.product.id}:${item.size.id}`;
            const existing = snapshot.get(key);
            if (existing) {
                existing.quantity += item.quantity;
            }
            else {
                snapshot.set(key, {
                    productId: item.product.id,
                    sizeId: item.size.id,
                    quantity: item.quantity,
                });
            }
        }
        return snapshot;
    }
    async applyStockDiff(originalSnapshot, updatedSnapshot, manager, referenceId) {
        for (const [key, updated] of updatedSnapshot.entries()) {
            const original = originalSnapshot.get(key);
            const originalQuantity = original?.quantity ?? 0;
            const difference = updated.quantity - originalQuantity;
            if (difference > 0) {
                await this.stockService.increaseStock({
                    productId: updated.productId,
                    sizeId: updated.sizeId,
                    quantity: difference,
                    movementType: "NEW",
                    manager,
                    referenceType: "PURCHASE_ORDER_UPDATE",
                    referenceId,
                });
            }
            else if (difference < 0) {
                await this.stockService.decreaseStock({
                    productId: updated.productId,
                    sizeId: updated.sizeId,
                    quantity: Math.abs(difference),
                    movementType: "SALE",
                    manager,
                    referenceType: "PURCHASE_ORDER_UPDATE",
                    referenceId,
                });
            }
            if (original) {
                originalSnapshot.delete(key);
            }
        }
        for (const [, original] of originalSnapshot.entries()) {
            await this.stockService.decreaseStock({
                productId: original.productId,
                sizeId: original.sizeId,
                quantity: original.quantity,
                movementType: "SALE",
                manager,
                referenceType: "PURCHASE_ORDER_UPDATE",
                referenceId,
            });
        }
    }
};
exports.PurchaseOrderService = PurchaseOrderService;
exports.PurchaseOrderService = PurchaseOrderService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(purchase_order_entity_1.PurchaseOrder)),
    __param(1, (0, typeorm_1.InjectRepository)(purchase_order_item_entity_1.PurchaseOrderItem)),
    __param(2, (0, typeorm_1.InjectRepository)(supplier_entity_1.Supplier)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.DataSource,
        stock_service_1.StockService])
], PurchaseOrderService);
//# sourceMappingURL=purchase-order.service.js.map