"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SupplierProductService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const supplier_product_entity_1 = require("../entities/supplier-product.entity");
const supplier_entity_1 = require("../entities/supplier.entity");
const product_entity_1 = require("../entities/product.entity");
const supplier_product_restock_log_entity_1 = require("../entities/supplier-product-restock-log.entity");
const stock_service_1 = require("../stock/stock.service");
const size_entity_1 = require("../entities/size.entity");
const supplier_product_size_entity_1 = require("../entities/supplier-product-size.entity");
const color_entity_1 = require("../entities/color.entity");
let SupplierProductService = class SupplierProductService {
    supplierProductRepository;
    supplierRepository;
    productRepository;
    sizeRepository;
    colorRepository;
    stockService;
    constructor(supplierProductRepository, supplierRepository, productRepository, sizeRepository, colorRepository, stockService) {
        this.supplierProductRepository = supplierProductRepository;
        this.supplierRepository = supplierRepository;
        this.productRepository = productRepository;
        this.sizeRepository = sizeRepository;
        this.colorRepository = colorRepository;
        this.stockService = stockService;
    }
    async assignProductToSupplier(createSupplierProductDto) {
        const { supplier_id, product_id, sizes } = createSupplierProductDto;
        const supplier = await this.supplierRepository.findOneBy({
            id: supplier_id,
        });
        if (!supplier) {
            throw new common_1.NotFoundException(`Supplier with ID ${supplier_id} not found`);
        }
        const product = await this.productRepository.findOneBy({ id: product_id });
        if (!product) {
            throw new common_1.NotFoundException(`Product with ID ${product_id} not found`);
        }
        const sizeRequests = this.extractSizeRequests(createSupplierProductDto);
        const resolvedSizes = await this.resolveSizes(sizeRequests, null);
        return this.createOrRestockSupplierProduct(supplier, product, {
            unit_price: createSupplierProductDto.unit_price,
            sizes: resolvedSizes,
            color: null,
            hasExplicitColor: false,
            status: createSupplierProductDto.status,
            note: createSupplierProductDto.note,
            purchase_date: createSupplierProductDto.purchase_date,
            requestedTotalQuantity: createSupplierProductDto.quantity,
        });
    }
    async createOrRestockSupplierProduct(supplier, product, payload) {
        return this.supplierProductRepository.manager.transaction(async (manager) => {
            const totalQuantity = payload.sizes.reduce((sum, entry) => sum + entry.quantity, 0);
            const derivedColor = this.derivePrimaryColor(payload.sizes);
            const recordColor = payload.hasExplicitColor
                ? payload.color ?? null
                : derivedColor;
            if (payload.requestedTotalQuantity !== undefined &&
                payload.requestedTotalQuantity !== totalQuantity) {
                throw new common_1.BadRequestException(`Provided quantity (${payload.requestedTotalQuantity}) does not match total of sizes (${totalQuantity}).`);
            }
            const supplierProductRepo = manager.getRepository(supplier_product_entity_1.SupplierProduct);
            const restockRepo = manager.getRepository(supplier_product_restock_log_entity_1.SupplierProductRestockLog);
            const supplierProductSizeRepo = manager.getRepository(supplier_product_size_entity_1.SupplierProductSize);
            const whereClause = {
                supplier: { id: supplier.id },
                product: { id: product.id },
                color_id: recordColor ? recordColor.id : undefined,
            };
            if (!recordColor) {
                whereClause.color_id = (0, typeorm_2.IsNull)();
            }
            let supplierProduct = await supplierProductRepo.findOne({
                where: whereClause,
                relations: {
                    sizes: { size: true, color: true },
                    color: true,
                },
            });
            if (!supplierProduct) {
                supplierProduct = supplierProductRepo.create({
                    supplier,
                    product,
                    unit_price: payload.unit_price,
                    quantity: totalQuantity,
                    color: recordColor ?? null,
                    color_id: recordColor?.id ?? null,
                    status: payload.status ?? "completed",
                    note: payload.note ?? null,
                    purchase_date: payload.purchase_date
                        ? new Date(payload.purchase_date)
                        : null,
                });
                supplierProduct.sizes = [];
            }
            else {
                supplierProduct.quantity += totalQuantity;
                supplierProduct.unit_price = payload.unit_price;
                supplierProduct.color = recordColor ?? null;
                supplierProduct.color_id = recordColor?.id ?? null;
                if (payload.status) {
                    supplierProduct.status = payload.status;
                }
                if (payload.note !== undefined) {
                    supplierProduct.note = payload.note;
                }
                if (payload.purchase_date !== undefined) {
                    supplierProduct.purchase_date = payload.purchase_date
                        ? new Date(payload.purchase_date)
                        : null;
                }
            }
            const savedSupplierProduct = await supplierProductRepo.save(supplierProduct);
            const existingSizeMap = new Map();
            if (supplierProduct.sizes) {
                for (const sizeEntry of supplierProduct.sizes) {
                    const key = this.buildVariantKey(sizeEntry.size_id, sizeEntry.color_id ?? null);
                    existingSizeMap.set(key, sizeEntry);
                }
            }
            for (const entry of payload.sizes) {
                const variantKey = this.buildVariantKey(entry.size.id, entry.color?.id ?? null);
                const existingSize = existingSizeMap.get(variantKey);
                if (existingSize) {
                    existingSize.quantity += entry.quantity;
                    await supplierProductSizeRepo.save(existingSize);
                }
                else {
                    const newSize = supplierProductSizeRepo.create({
                        supplierProduct: savedSupplierProduct,
                        supplier_product_id: savedSupplierProduct.id,
                        size: entry.size,
                        size_id: entry.size.id,
                        quantity: entry.quantity,
                        color: entry.color ?? null,
                        color_id: entry.color?.id ?? null,
                    });
                    const savedSize = await supplierProductSizeRepo.save(newSize);
                    existingSizeMap.set(variantKey, savedSize);
                }
                await this.stockService.increaseStock({
                    productId: product.id,
                    sizeId: entry.size.id,
                    quantity: entry.quantity,
                    colorId: entry.color?.id ?? recordColor?.id ?? null,
                    movementType: "NEW",
                    manager,
                    referenceType: "SUPPLIER_PRODUCT",
                    referenceId: savedSupplierProduct.id,
                    createIfMissing: true,
                });
            }
            const restockLog = restockRepo.create({
                supplierProduct: savedSupplierProduct,
                quantity: totalQuantity,
                unit_price: payload.unit_price,
            });
            await restockRepo.save(restockLog);
            return supplierProductRepo.findOneOrFail({
                where: { id: savedSupplierProduct.id },
                relations: {
                    product: true,
                    supplier: true,
                    color: true,
                    sizes: { size: true, color: true },
                    restockLogs: true,
                },
                order: {
                    restockLogs: { restocked_at: "DESC" },
                },
            });
        });
    }
    async resolveSizes(sizes, defaultColorId) {
        const uniqueIds = Array.from(new Set(sizes.map((item) => item.size_id)));
        const sizeEntities = await this.sizeRepository.findBy({
            id: (0, typeorm_2.In)(uniqueIds),
        });
        if (sizeEntities.length !== uniqueIds.length) {
            const foundIds = new Set(sizeEntities.map((size) => size.id));
            const missingId = uniqueIds.find((id) => !foundIds.has(id));
            throw new common_1.NotFoundException(`Size with ID ${missingId ?? "unknown"} not found`);
        }
        const sizeMap = new Map(sizeEntities.map((size) => [size.id, size]));
        const colorIds = new Set();
        if (defaultColorId !== null) {
            colorIds.add(defaultColorId);
        }
        for (const sizeDto of sizes) {
            if (sizeDto.color_id !== undefined && sizeDto.color_id !== null) {
                colorIds.add(sizeDto.color_id);
            }
        }
        const colorEntities = colorIds.size
            ? await this.colorRepository.find({
                where: { id: (0, typeorm_2.In)(Array.from(colorIds)) },
            })
            : [];
        if (colorIds.size && colorEntities.length !== colorIds.size) {
            const resolvedIds = new Set(colorEntities.map((color) => color.id));
            const missing = Array.from(colorIds).filter((id) => !resolvedIds.has(id));
            throw new common_1.NotFoundException(`Colors not found for identifiers: ${missing.join(", ")}`);
        }
        const colorMap = new Map(colorEntities.map((color) => [color.id, color]));
        const defaultColor = defaultColorId !== null ? colorMap.get(defaultColorId) ?? null : null;
        const aggregated = new Map();
        for (const sizeDto of sizes) {
            const size = sizeMap.get(sizeDto.size_id);
            if (!size) {
                throw new common_1.NotFoundException(`Size with ID ${sizeDto.size_id} not found`);
            }
            let resolvedColor = defaultColor ?? null;
            if (sizeDto.color_id !== undefined) {
                if (sizeDto.color_id === null) {
                    resolvedColor = null;
                }
                else {
                    const color = colorMap.get(sizeDto.color_id);
                    if (!color) {
                        throw new common_1.NotFoundException(`Color with ID ${sizeDto.color_id} not found`);
                    }
                    resolvedColor = color;
                }
            }
            const key = this.buildVariantKey(size.id, resolvedColor?.id ?? null);
            const existing = aggregated.get(key);
            if (existing) {
                existing.quantity += sizeDto.quantity;
            }
            else {
                aggregated.set(key, {
                    size,
                    quantity: sizeDto.quantity,
                    color: resolvedColor,
                });
            }
        }
        return Array.from(aggregated.values());
    }
    async searchSupplierProducts(query) {
        const page = Math.max(Number.parseInt(query.page ?? "1", 10) || 1, 1);
        const limit = Math.max(Number.parseInt(query.limit ?? "10", 10) || 10, 1);
        const search = query.searchTerm.toLowerCase();
        const supplierId = query.supplierId
            ? Number.parseInt(query.supplierId, 10)
            : undefined;
        if (supplierId !== undefined && (Number.isNaN(supplierId) || supplierId <= 0)) {
            throw new common_1.BadRequestException("supplierId must be a positive number when provided");
        }
        const queryBuilder = this.supplierProductRepository
            .createQueryBuilder("supplierProduct")
            .leftJoinAndSelect("supplierProduct.product", "product")
            .leftJoinAndSelect("supplierProduct.supplier", "supplier")
            .leftJoinAndSelect("supplierProduct.color", "color")
            .leftJoinAndSelect("supplierProduct.sizes", "variant")
            .leftJoinAndSelect("variant.size", "size")
            .leftJoinAndSelect("variant.color", "variantColor")
            .leftJoinAndSelect("supplierProduct.restockLogs", "restockLogs")
            .orderBy("supplierProduct.created_at", "DESC")
            .addOrderBy("supplierProduct.id", "DESC")
            .distinct(true);
        if (supplierId !== undefined) {
            queryBuilder.andWhere("supplierProduct.supplier_id = :supplierId", {
                supplierId,
            });
        }
        if (search) {
            queryBuilder.andWhere("(LOWER(product.product_name) LIKE :search OR LOWER(product.product_code) LIKE :search OR LOWER(product.product_color) LIKE :search OR LOWER(supplier.name) LIKE :search)", { search: `%${search}%` });
        }
        const [data, total] = await queryBuilder
            .skip((page - 1) * limit)
            .take(limit)
            .getManyAndCount();
        const totalPages = total > 0 ? Math.ceil(total / limit) : 0;
        return {
            data,
            page,
            limit,
            total,
            previous: page > 1 ? page - 1 : null,
            next: page < totalPages ? page + 1 : null,
        };
    }
    async getProductVariantOptions(productId) {
        const product = await this.productRepository.findOne({
            where: { id: productId },
            relations: { sizes: true, colors: true },
        });
        if (!product) {
            throw new common_1.NotFoundException(`Product with ID ${productId} not found`);
        }
        const sizeMap = new Map();
        for (const size of product.sizes ?? []) {
            sizeMap.set(size.id, size);
        }
        const colorMap = new Map();
        for (const color of product.colors ?? []) {
            colorMap.set(color.id, color);
        }
        const supplierVariants = await this.supplierProductRepository.find({
            where: { product: { id: productId } },
            relations: {
                sizes: { size: true, color: true },
                color: true,
            },
        });
        for (const variant of supplierVariants) {
            if (variant.color) {
                colorMap.set(variant.color.id, variant.color);
            }
            for (const sizeEntry of variant.sizes ?? []) {
                if (sizeEntry.size) {
                    sizeMap.set(sizeEntry.size.id, sizeEntry.size);
                }
                if (sizeEntry.color) {
                    colorMap.set(sizeEntry.color.id, sizeEntry.color);
                }
            }
        }
        const sizes = Array.from(sizeMap.values()).sort((a, b) => a.name.localeCompare(b.name));
        const colors = Array.from(colorMap.values()).sort((a, b) => a.name.localeCompare(b.name));
        return {
            productId: product.id,
            productName: product.product_name,
            sizes: sizes.map((size) => ({ id: size.id, name: size.name })),
            colors: colors.map((color) => ({ id: color.id, name: color.name })),
        };
    }
    async findProductsBySupplier(supplierId) {
        const supplier = await this.supplierRepository.findOneBy({
            id: supplierId,
        });
        if (!supplier) {
            throw new common_1.NotFoundException(`Supplier with ID ${supplierId} not found`);
        }
        return this.supplierProductRepository.find({
            where: { supplier: { id: supplierId } },
            relations: {
                product: true,
                supplier: true,
                color: true,
                sizes: { size: true, color: true },
                restockLogs: true,
            },
            order: {
                created_at: "DESC",
                restockLogs: { restocked_at: "DESC" },
            },
        });
    }
    async removeAssignment(supplierId, productId) {
        const result = await this.supplierProductRepository.delete({
            supplier: { id: supplierId },
            product: { id: productId },
        });
        if (result.affected === 0) {
            throw new common_1.NotFoundException(`Assignment of product ${productId} to supplier ${supplierId} not found`);
        }
    }
    async createBulk(createBulkSupplierProductDto) {
        const { supplier_id, products } = createBulkSupplierProductDto;
        const supplier = await this.supplierRepository.findOneBy({
            id: supplier_id,
        });
        if (!supplier) {
            throw new common_1.NotFoundException(`Supplier with ID ${supplier_id} not found`);
        }
        const supplierProductsToCreate = [];
        for (const productDto of products) {
            const product = await this.productRepository.findOneBy({
                id: productDto.product_id,
            });
            if (!product) {
                throw new common_1.NotFoundException(`Product with ID ${productDto.product_id} not found`);
            }
            const sizeRequests = this.extractSizeRequests(productDto);
            const resolvedSizes = await this.resolveSizes(sizeRequests, null);
            const existingSupplierProduct = await this.createOrRestockSupplierProduct(supplier, product, {
                unit_price: productDto.unit_price,
                sizes: resolvedSizes,
                color: null,
                hasExplicitColor: false,
                status: productDto.status,
                note: productDto.note,
                purchase_date: productDto.purchase_date,
                requestedTotalQuantity: productDto.quantity,
            });
            supplierProductsToCreate.push(existingSupplierProduct);
        }
        return supplierProductsToCreate;
    }
    extractSizeRequests(dto) {
        const entries = [];
        if (Array.isArray(dto.sizes)) {
            for (const item of dto.sizes) {
                entries.push({
                    size_id: item.size_id,
                    quantity: item.quantity,
                    color_id: item.color_id,
                });
            }
        }
        if (entries.length === 0) {
            throw new common_1.BadRequestException("At least one size entry is required.");
        }
        return entries;
    }
    buildVariantKey(sizeId, colorId) {
        return `${sizeId}::${colorId ?? "null"}`;
    }
    derivePrimaryColor(sizes) {
        const colorIds = new Set();
        let firstColor = null;
        let hasNull = false;
        for (const entry of sizes) {
            if (!entry.color) {
                hasNull = true;
                continue;
            }
            colorIds.add(entry.color.id);
            if (!firstColor) {
                firstColor = entry.color;
            }
            else if (entry.color.id !== firstColor.id) {
                return null;
            }
        }
        if (hasNull) {
            return null;
        }
        if (colorIds.size === 1 && firstColor) {
            return firstColor;
        }
        return null;
    }
};
exports.SupplierProductService = SupplierProductService;
exports.SupplierProductService = SupplierProductService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(supplier_product_entity_1.SupplierProduct)),
    __param(1, (0, typeorm_1.InjectRepository)(supplier_entity_1.Supplier)),
    __param(2, (0, typeorm_1.InjectRepository)(product_entity_1.Product)),
    __param(3, (0, typeorm_1.InjectRepository)(size_entity_1.Size)),
    __param(4, (0, typeorm_1.InjectRepository)(color_entity_1.Color)),
    __metadata("design:paramtypes", [typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        typeorm_2.Repository,
        stock_service_1.StockService])
], SupplierProductService);
//# sourceMappingURL=supplier-product.service.js.map