import { Injectable, UnauthorizedException } from "@nestjs/common";
import { ConfigService } from "@nestjs/config";
import { PassportStrategy } from "@nestjs/passport";
import { ExtractJwt, Strategy } from "passport-jwt";
import { UsersService } from "../../users/users.service";
import { User } from "../../entities/user.entity";

interface JwtPayload {
  sub: number;
  email: string;
}

@Injectable()
export class JwtStrategy extends PassportStrategy(Strategy) {
  constructor(
    configService: ConfigService,
    private readonly usersService: UsersService,
  ) {
    super({
      jwtFromRequest: ExtractJwt.fromAuthHeaderAsBearerToken(),
      ignoreExpiration: false,
      secretOrKey: configService.get<string>("JWT_SECRET", "change-me"),
    });
  }

  async validate(payload: JwtPayload): Promise<Omit<User, "password">> {
    const user = await this.usersService.findById(payload.sub);
    if (!user || user.status !== 1) {
      throw new UnauthorizedException("Invalid token");
    }
    const { password, ...userWithoutPassword } = user;
    return userWithoutPassword as Omit<User, "password">;
  }
}
