import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Patch,
  Post,
  Query,
} from "@nestjs/common";
import {
  ApiBody,
  ApiOperation,
  ApiQuery,
  ApiParam,
  ApiResponse,
  ApiTags,
} from "@nestjs/swagger";
import { PaginationDto } from "../common/dto/pagination.dto";
import { CategoryService } from "./category.service";
import { CreateCategoryDto } from "./dto/create-category.dto";
import { UpdateCategoryDto } from "./dto/update-category.dto";

@ApiTags("Categories")
@Controller("categories")
export class CategoryController {
  constructor(private readonly categoryService: CategoryService) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a new category" })
  @ApiResponse({
    status: 201,
    description: "The category has been successfully created.",
  })
  @ApiResponse({ status: 400, description: "Bad Request." })
  create(@Body() createCategoryDto: CreateCategoryDto) {
    return this.categoryService.create(createCategoryDto);
  }

  @Get()
  @ApiOperation({
    summary: "Retrieve categories with pagination, relations, and optional search",
  })
  @ApiResponse({
    status: 200,
    description: "Returns all categories including parent, children, and products.",
  })
  @ApiQuery({
    name: "limit",
    type: Number,
    required: false,
    description: "Limit of items per page",
  })
  @ApiQuery({
    name: "page",
    type: Number,
    required: false,
    description: "Page number to retrieve",
  })
  @ApiQuery({
    name: "search",
    type: String,
    required: false,
    description: "Search keyword applied to category name, title, and subtitle",
  })
  findAll(@Query() paginationDto: PaginationDto) {
    return this.categoryService.findAll(paginationDto);
  }

  @Get(":id")
  @ApiOperation({ summary: "Retrieve a category by ID" })
  @ApiParam({ name: "id", description: "ID of the category to retrieve" })
  @ApiResponse({
    status: 200,
    description: "Returns the category with the specified ID.",
  })
  @ApiResponse({ status: 404, description: "Category not found." })
  findOne(@Param("id") id: string) {
    return this.categoryService.findOne(+id);
  }

  @Patch(":id")
  @ApiOperation({ summary: "Update an existing category" })
  @ApiParam({ name: "id", description: "ID of the category to update" })
  @ApiBody({
    type: UpdateCategoryDto,
    description: "Fields to update on the category",
  })
  @ApiResponse({
    status: 200,
    description: "The category has been successfully updated.",
  })
  @ApiResponse({ status: 404, description: "Category not found." })
  @ApiResponse({ status: 400, description: "Bad Request." })
  update(@Param("id") id: string, @Body() updateCategoryDto: UpdateCategoryDto) {
    return this.categoryService.update(+id, updateCategoryDto);
  }

  @Delete(":id")
  @HttpCode(HttpStatus.NO_CONTENT)
  @ApiOperation({ summary: "Delete a category by ID" })
  @ApiParam({ name: "id", description: "ID of the category to delete" })
  @ApiResponse({
    status: 204,
    description: "The category has been successfully deleted.",
  })
  @ApiResponse({ status: 404, description: "Category not found." })
  async remove(@Param("id") id: string) {
    await this.categoryService.remove(+id);
  }
}
