import {
  BadRequestException,
  Injectable,
  NotFoundException,
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { In, Repository } from "typeorm";
import { Category } from "../entities/category.entity";
import { PaginationDto } from "../common/dto/pagination.dto";
import { CreateCategoryDto } from "./dto/create-category.dto";
import { UpdateCategoryDto } from "./dto/update-category.dto";

@Injectable()
export class CategoryService {
  constructor(
    @InjectRepository(Category)
    private readonly categoryRepository: Repository<Category>,
  ) {}

  async create(createCategoryDto: CreateCategoryDto): Promise<Category> {
    const { parentId, status, ...categoryData } = createCategoryDto;

    let parent: Category | null = null;
    if (parentId !== undefined && parentId !== null) {
      parent = await this.findExistingParent(parentId);
    }

    const category = this.categoryRepository.create({
      ...categoryData,
      status: status ?? 1,
      parent,
    });
    return this.categoryRepository.save(category);
  }

  async findAll(
    paginationDto: PaginationDto,
  ): Promise<{
    data: Category[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(
      Number.parseInt(paginationDto.page ?? "1", 10) || 1,
      1,
    );
    const limit = Math.max(
      Number.parseInt(paginationDto.limit ?? "10", 10) || 10,
      1,
    );
    const search = paginationDto.search?.trim() ?? "";

    const queryBuilder =
      this.categoryRepository.createQueryBuilder("category");

    if (search) {
      queryBuilder.where(
        "(LOWER(category.category_name) LIKE LOWER(:search) OR LOWER(category.title) LIKE LOWER(:search) OR LOWER(category.subtitle) LIKE LOWER(:search))",
        { search: `%${search}%` },
      );
    }

    queryBuilder.orderBy("category.category_name", "ASC");

    const total = await queryBuilder.clone().getCount();

    const rawIds = await queryBuilder
      .clone()
      .select("category.id", "id")
      .skip((page - 1) * limit)
      .take(limit)
      .getRawMany<{ id: number }>();

    const ids = rawIds.map((row) => Number(row.id)).filter(Boolean);

    let data: Category[] = [];

    if (ids.length > 0) {
      const categories = await this.categoryRepository.find({
        where: { id: In(ids) },
        relations: ["parent", "children", "products"],
        order: { category_name: "ASC" },
      });

      const categoriesById = new Map(
        categories.map((category) => [category.id, category]),
      );

      data = ids
        .map((id) => categoriesById.get(id))
        .filter((category): category is Category => Boolean(category));
    }

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async findOne(id: number): Promise<Category> {
    const category = await this.categoryRepository.findOne({
      where: { id },
      relations: ["parent", "children", "products"],
    });
    if (!category) {
      throw new NotFoundException(`Category with ID ${id} not found`);
    }
    return category;
  }

  async update(
    id: number,
    updateCategoryDto: UpdateCategoryDto,
  ): Promise<Category> {
    const category = await this.findOne(id);
    const { parentId, status, ...categoryData } = updateCategoryDto;

    if (parentId !== undefined) {
      if (parentId === null) {
        category.parent = null;
      } else {
        if (parentId === id) {
          throw new BadRequestException("A category cannot be its own parent");
        }
        category.parent = await this.findExistingParent(parentId);
      }
    }

    if (status !== undefined) {
      category.status = status;
    }

    this.categoryRepository.merge(category, categoryData);
    return this.categoryRepository.save(category);
  }

  async remove(id: number): Promise<void> {
    const category = await this.findOne(id);
    await this.categoryRepository.remove(category);
  }

  private async findExistingParent(parentId: number): Promise<Category> {
    const parent = await this.categoryRepository.findOne({
      where: { id: parentId },
    });

    if (!parent) {
      throw new NotFoundException(`Parent category with ID ${parentId} not found`);
    }
    return parent;
  }
}
