import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Patch,
  Post,
  Query,
} from "@nestjs/common";
import {
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiTags,
} from "@nestjs/swagger";
import { PaginationDto } from "../common/dto/pagination.dto";
import { ColorService } from "./color.service";
import { CreateColorDto } from "./dto/create-color.dto";
import { UpdateColorDto } from "./dto/update-color.dto";

@ApiTags("Colors")
@Controller("colors")
export class ColorController {
  constructor(private readonly colorService: ColorService) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a new color" })
  @ApiResponse({
    status: 201,
    description: "The color has been created.",
  })
  @ApiResponse({ status: 400, description: "Validation failed." })
  create(@Body() createColorDto: CreateColorDto) {
    return this.colorService.create(createColorDto);
  }

  @Get()
  @ApiOperation({ summary: "Retrieve colors with pagination and search" })
  @ApiResponse({
    status: 200,
    description: "Paginated list of colors.",
  })
  @ApiQuery({
    name: "limit",
    type: Number,
    required: false,
    description: "Number of items per page",
  })
  @ApiQuery({
    name: "page",
    type: Number,
    required: false,
    description: "Page number",
  })
  @ApiQuery({
    name: "query",
    type: String,
    required: false,
    description: "Search keyword (alias: search)",
  })
  @ApiQuery({
    name: "search",
    type: String,
    required: false,
    description: "Deprecated search keyword; prefer query",
  })
  findAll(@Query() paginationDto: PaginationDto) {
    return this.colorService.findAll(paginationDto);
  }

  @Get(":id")
  @ApiOperation({ summary: "Retrieve a color by ID" })
  @ApiParam({ name: "id", description: "Color identifier" })
  @ApiResponse({
    status: 200,
    description: "Returns the color with the specified ID.",
  })
  @ApiResponse({ status: 404, description: "Color not found." })
  findOne(@Param("id") id: string) {
    return this.colorService.findOne(+id);
  }

  @Patch(":id")
  @ApiOperation({ summary: "Update an existing color" })
  @ApiParam({ name: "id", description: "Color identifier" })
  @ApiBody({ type: UpdateColorDto })
  @ApiResponse({
    status: 200,
    description: "The color has been updated.",
  })
  @ApiResponse({ status: 404, description: "Color not found." })
  update(@Param("id") id: string, @Body() updateColorDto: UpdateColorDto) {
    return this.colorService.update(+id, updateColorDto);
  }

  @Delete(":id")
  @HttpCode(HttpStatus.NO_CONTENT)
  @ApiOperation({ summary: "Delete a color" })
  @ApiParam({ name: "id", description: "Color identifier" })
  @ApiResponse({ status: 204, description: "Color removed." })
  @ApiResponse({ status: 404, description: "Color not found." })
  async remove(@Param("id") id: string) {
    await this.colorService.remove(+id);
  }
}
