import {
  BadRequestException,
  Injectable,
  NotFoundException,
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { In, Repository } from "typeorm";
import { Color } from "../entities/color.entity";
import { PaginationDto } from "../common/dto/pagination.dto";
import { CreateColorDto } from "./dto/create-color.dto";
import { UpdateColorDto } from "./dto/update-color.dto";

@Injectable()
export class ColorService {
  constructor(
    @InjectRepository(Color)
    private readonly colorRepository: Repository<Color>
  ) {}

  async create(createColorDto: CreateColorDto): Promise<Color> {
    const color = this.colorRepository.create(createColorDto);
    return this.colorRepository.save(color);
  }

  async findAll(
    paginationDto: PaginationDto
  ): Promise<{
    data: Color[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(
      Number.parseInt(paginationDto.page ?? "1", 10) || 1,
      1
    );
    const limit = Math.max(
      Number.parseInt(paginationDto.limit ?? "10", 10) || 10,
      1
    );
    const search = paginationDto.search?.trim().toLowerCase() ?? "";

    const queryBuilder = this.colorRepository
      .createQueryBuilder("color")
      .orderBy("color.created_at", "DESC");

    if (search) {
      queryBuilder.where("LOWER(color.name) LIKE :search", {
        search: `%${search}%`,
      });
    }

    const [data, total] = await queryBuilder
      .skip((page - 1) * limit)
      .take(limit)
      .getManyAndCount();

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async findOne(id: number): Promise<Color> {
    const color = await this.colorRepository.findOne({ where: { id } });
    if (!color) {
      throw new NotFoundException(`Color with ID ${id} not found`);
    }
    return color;
  }

  async update(id: number, updateColorDto: UpdateColorDto): Promise<Color> {
    const color = await this.findOne(id);
    this.colorRepository.merge(color, updateColorDto);
    return this.colorRepository.save(color);
  }

  async remove(id: number): Promise<void> {
    const result = await this.colorRepository.delete(id);
    if (result.affected === 0) {
      throw new NotFoundException(`Color with ID ${id} not found`);
    }
  }

  async resolveColors(colorIds: number[]): Promise<Map<number, Color>> {
    const uniqueIds = Array.from(new Set(colorIds));
    if (uniqueIds.length === 0) {
      throw new BadRequestException("At least one color must be provided");
    }

    const colors = await this.colorRepository.find({
      where: { id: In(uniqueIds) },
    });

    if (colors.length !== uniqueIds.length) {
      const foundIds = new Set(colors.map((color) => color.id));
      const missing = uniqueIds.filter((id) => !foundIds.has(id));
      throw new NotFoundException(
        `Colors not found for identifiers: ${missing.join(", ")}`
      );
    }

    return new Map(colors.map((color) => [color.id, color]));
  }
}
