import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Post,
  Put,
  Query,
} from "@nestjs/common";
import {
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiTags,
} from "@nestjs/swagger";
import { CustomerService } from "./customer.service";
import { CreateCustomerDto } from "./dto/create-customer.dto";
import { UpdateCustomerDto } from "./dto/update-customer.dto";
import { PaginationDto } from "../common/dto/pagination.dto";

@ApiTags("Customers")
@Controller("customers")
export class CustomerController {
  constructor(private readonly customerService: CustomerService) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a new customer" })
  @ApiResponse({
    status: 201,
    description: "The customer has been successfully created.",
  })
  @ApiResponse({ status: 400, description: "Validation failed." })
  create(@Body() createCustomerDto: CreateCustomerDto) {
    return this.customerService.create(createCustomerDto);
  }

  @Get()
  @ApiOperation({
    summary: "Retrieve customers with pagination and optional search",
  })
  @ApiResponse({
    status: 200,
    description: "Returns customers with pagination metadata.",
  })
  @ApiResponse({ status: 400, description: "Validation failed." })
  @ApiQuery({
    name: "limit",
    type: Number,
    required: false,
    description: "Limit of items per page",
  })
  @ApiQuery({
    name: "page",
    type: Number,
    required: false,
    description: "Page of items",
  })
  @ApiQuery({
    name: "query",
    type: String,
    required: false,
    description: "Search keyword (alias: search)",
  })
  @ApiQuery({
    name: "search",
    type: String,
    required: false,
    description: "Deprecated search keyword; prefer query",
  })
  findAll(@Query() paginationDto: PaginationDto) {
    return this.customerService.findAll(paginationDto);
  }

  @Get(":id")
  @ApiOperation({ summary: "Retrieve a customer by ID" })
  @ApiParam({ name: "id", description: "Customer identifier" })
  @ApiResponse({
    status: 200,
    description: "Returns the customer with the specified ID.",
  })
  @ApiResponse({ status: 404, description: "Customer not found." })
  findOne(@Param("id") id: string) {
    return this.customerService.findOne(+id);
  }

  @Put(":id")
  @ApiOperation({ summary: "Update an existing customer" })
  @ApiParam({ name: "id", description: "Customer identifier" })
  @ApiBody({ type: UpdateCustomerDto })
  @ApiResponse({
    status: 200,
    description: "The customer has been successfully updated.",
  })
  @ApiResponse({ status: 404, description: "Customer not found." })
  update(@Param("id") id: string, @Body() updateCustomerDto: UpdateCustomerDto) {
    return this.customerService.update(+id, updateCustomerDto);
  }

  @Delete(":id")
  @HttpCode(HttpStatus.NO_CONTENT)
  @ApiOperation({ summary: "Delete a customer" })
  @ApiParam({ name: "id", description: "Customer identifier" })
  @ApiResponse({ status: 204, description: "Customer removed." })
  @ApiResponse({ status: 404, description: "Customer not found." })
  remove(@Param("id") id: string) {
    return this.customerService.remove(+id);
  }
}
