import { Injectable, NotFoundException } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { Repository } from "typeorm";
import { Customer } from "../entities/customer.entity";
import { CreateCustomerDto } from "./dto/create-customer.dto";
import { UpdateCustomerDto } from "./dto/update-customer.dto";
import { PaginationDto } from "../common/dto/pagination.dto";

@Injectable()
export class CustomerService {
  constructor(
    @InjectRepository(Customer)
    private readonly customerRepository: Repository<Customer>
  ) {}

  create(createCustomerDto: CreateCustomerDto): Promise<Customer> {
    const customer = this.customerRepository.create({
      first_name: createCustomerDto.firstName,
      last_name: createCustomerDto.lastName,
      email: createCustomerDto.email,
      phone: createCustomerDto.phone ?? null,
      address: createCustomerDto.address ?? null,
    });
    return this.customerRepository.save(customer);
  }

  async findAll(
    paginationDto: PaginationDto,
  ): Promise<{
    data: Customer[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(
      Number.parseInt(paginationDto.page ?? "1", 10) || 1,
      1,
    );
    const limit = Math.max(
      Number.parseInt(paginationDto.limit ?? "10", 10) || 10,
      1,
    );
    const search = paginationDto.searchTerm;

    const queryBuilder = this.customerRepository
      .createQueryBuilder("customer")
      .orderBy("customer.created_at", "DESC");

    if (search) {
      queryBuilder.where(
        "(LOWER(customer.first_name) LIKE LOWER(:search) OR LOWER(customer.last_name) LIKE LOWER(:search) OR LOWER(customer.email) LIKE LOWER(:search) OR LOWER(customer.phone) LIKE LOWER(:search))",
        { search: `%${search}%` },
      );
    }

    const [data, total] = await queryBuilder
      .skip((page - 1) * limit)
      .take(limit)
      .getManyAndCount();

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async findOne(id: number): Promise<Customer> {
    const customer = await this.customerRepository.findOne({ where: { id } });
    if (!customer) {
      throw new NotFoundException(`Customer with ID ${id} not found`);
    }
    return customer;
  }

  async update(
    id: number,
    updateCustomerDto: UpdateCustomerDto
  ): Promise<Customer> {
    const customer = await this.findOne(id);
    customer.first_name =
      updateCustomerDto.firstName ?? customer.first_name;
    customer.last_name = updateCustomerDto.lastName ?? customer.last_name;
    customer.email = updateCustomerDto.email ?? customer.email;
    customer.phone =
      updateCustomerDto.phone !== undefined
        ? updateCustomerDto.phone
        : customer.phone;
    customer.address =
      updateCustomerDto.address !== undefined
        ? updateCustomerDto.address
        : customer.address;
    return this.customerRepository.save(customer);
  }

  async remove(id: number): Promise<void> {
    const result = await this.customerRepository.delete(id);
    if (result.affected === 0) {
      throw new NotFoundException(`Customer with ID ${id} not found`);
    }
  }
}
