import { BadRequestException } from "@nestjs/common";
import { Test, TestingModule } from "@nestjs/testing";
import { getRepositoryToken } from "@nestjs/typeorm";
import { DashboardService } from "./dashboard.service";
import { SupplierProduct } from "../entities/supplier-product.entity";
import { Order } from "../entities/order.entity";
import { Customer } from "../entities/customer.entity";
import { OrderReturn } from "../entities/order-return.entity";

const createMockRepository = () => ({
  count: jest.fn(),
});

type MockRepository = ReturnType<typeof createMockRepository>;

describe("DashboardService", () => {
  let service: DashboardService;
  let supplierProductRepository: MockRepository;
  let orderRepository: MockRepository;
  let customerRepository: MockRepository;
  let orderReturnRepository: MockRepository;

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        DashboardService,
        {
          provide: getRepositoryToken(SupplierProduct),
          useValue: createMockRepository(),
        },
        {
          provide: getRepositoryToken(Order),
          useValue: createMockRepository(),
        },
        {
          provide: getRepositoryToken(Customer),
          useValue: createMockRepository(),
        },
        {
          provide: getRepositoryToken(OrderReturn),
          useValue: createMockRepository(),
        },
      ],
    }).compile();

    service = module.get<DashboardService>(DashboardService);
    supplierProductRepository = module.get<MockRepository>(
      getRepositoryToken(SupplierProduct),
    );
    orderRepository = module.get<MockRepository>(getRepositoryToken(Order));
    customerRepository = module.get<MockRepository>(
      getRepositoryToken(Customer),
    );
    orderReturnRepository = module.get<MockRepository>(
      getRepositoryToken(OrderReturn),
    );
  });

  it("returns aggregated overview totals without filters", async () => {
    supplierProductRepository.count.mockResolvedValue(42);
    orderRepository.count.mockResolvedValue(15);
    customerRepository.count.mockResolvedValue(8);
    orderReturnRepository.count.mockResolvedValue(3);

    await expect(service.getOverview()).resolves.toEqual({
      totalSupplierProducts: 42,
      totalOrders: 15,
      totalCustomers: 8,
      totalReturns: 3,
    });

    expect(supplierProductRepository.count).toHaveBeenCalledWith(undefined);
    expect(orderRepository.count).toHaveBeenCalledWith(undefined);
    expect(customerRepository.count).toHaveBeenCalledWith(undefined);
    expect(orderReturnRepository.count).toHaveBeenCalledWith(undefined);
  });

  it("applies date filters when provided", async () => {
    supplierProductRepository.count.mockResolvedValue(0);
    orderRepository.count.mockResolvedValue(0);
    customerRepository.count.mockResolvedValue(0);
    orderReturnRepository.count.mockResolvedValue(0);

    const filters = {
      startDate: "2024-01-01T00:00:00.000Z",
      endDate: "2024-01-31T23:59:59.999Z",
    };

    await service.getOverview(filters);

    const expectedWhere = expect.objectContaining({
      created_at: expect.any(Object),
    });

    expect(supplierProductRepository.count).toHaveBeenCalledWith(
      expect.objectContaining({ where: expectedWhere }),
    );
    expect(orderRepository.count).toHaveBeenCalledWith(
      expect.objectContaining({ where: expectedWhere }),
    );
    expect(customerRepository.count).toHaveBeenCalledWith(
      expect.objectContaining({ where: expectedWhere }),
    );
    expect(orderReturnRepository.count).toHaveBeenCalledWith(
      expect.objectContaining({ where: expectedWhere }),
    );
  });

  it("rejects when startDate is after endDate", async () => {
    await expect(
      service.getOverview({
        startDate: "2024-02-15T00:00:00.000Z",
        endDate: "2024-02-10T00:00:00.000Z",
      }),
    ).rejects.toBeInstanceOf(BadRequestException);

    expect(supplierProductRepository.count).not.toHaveBeenCalled();
    expect(orderRepository.count).not.toHaveBeenCalled();
    expect(customerRepository.count).not.toHaveBeenCalled();
    expect(orderReturnRepository.count).not.toHaveBeenCalled();
  });
});
