import { BadRequestException, Injectable } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import {
  Between,
  FindOperator,
  FindOptionsWhere,
  LessThanOrEqual,
  MoreThanOrEqual,
  Repository,
} from "typeorm";
import { SupplierProduct } from "../entities/supplier-product.entity";
import { Order } from "../entities/order.entity";
import { Customer } from "../entities/customer.entity";
import { OrderReturn } from "../entities/order-return.entity";
import { DashboardOverviewDto } from "./dto/dashboard-overview.dto";
import { DashboardOverviewQueryDto } from "./dto/dashboard-overview-query.dto";

@Injectable()
export class DashboardService {
  constructor(
    @InjectRepository(SupplierProduct)
    private readonly supplierProductRepository: Repository<SupplierProduct>,
    @InjectRepository(Order)
    private readonly orderRepository: Repository<Order>,
    @InjectRepository(Customer)
    private readonly customerRepository: Repository<Customer>,
    @InjectRepository(OrderReturn)
    private readonly orderReturnRepository: Repository<OrderReturn>,
  ) {}

  async getOverview(
    filters?: DashboardOverviewQueryDto,
  ): Promise<DashboardOverviewDto> {
    const dateRange = this.parseDateRange(filters);

    const supplierProductWhere = this.buildDateWhere<SupplierProduct>(
      "created_at",
      dateRange,
    );
    const orderWhere = this.buildDateWhere<Order>("created_at", dateRange);
    const customerWhere = this.buildDateWhere<Customer>(
      "created_at",
      dateRange,
    );
    const orderReturnWhere = this.buildDateWhere<OrderReturn>(
      "created_at",
      dateRange,
    );

    const [
      totalSupplierProducts,
      totalOrders,
      totalCustomers,
      totalReturns,
    ] = await Promise.all([
      this.supplierProductRepository.count(
        supplierProductWhere ? { where: supplierProductWhere } : undefined,
      ),
      this.orderRepository.count(orderWhere ? { where: orderWhere } : undefined),
      this.customerRepository.count(
        customerWhere ? { where: customerWhere } : undefined,
      ),
      this.orderReturnRepository.count(
        orderReturnWhere ? { where: orderReturnWhere } : undefined,
      ),
    ]);

    return {
      totalSupplierProducts,
      totalOrders,
      totalCustomers,
      totalReturns,
    };
  }

  private parseDateRange(
    filters?: DashboardOverviewQueryDto,
  ): { start?: Date; end?: Date } {
    if (!filters?.startDate && !filters?.endDate) {
      return {};
    }

    const start = filters.startDate ? new Date(filters.startDate) : undefined;
    const end = filters.endDate ? new Date(filters.endDate) : undefined;

    if (start && Number.isNaN(start.getTime())) {
      throw new BadRequestException("startDate must be a valid ISO date");
    }

    if (end && Number.isNaN(end.getTime())) {
      throw new BadRequestException("endDate must be a valid ISO date");
    }

    if (start && end && start > end) {
      throw new BadRequestException("startDate must be before endDate");
    }

    return { start, end };
  }

  private buildDateWhere<T>(
    field: keyof T & string,
    range: { start?: Date; end?: Date },
  ): FindOptionsWhere<T> | undefined {
    const condition = this.createDateCondition(range.start, range.end);
    if (!condition) {
      return undefined;
    }

    return { [field]: condition } as FindOptionsWhere<T>;
  }

  private createDateCondition(
    start?: Date,
    end?: Date,
  ): FindOperator<Date> | undefined {
    if (start && end) {
      return Between(start, end);
    }
    if (start) {
      return MoreThanOrEqual(start);
    }
    if (end) {
      return LessThanOrEqual(end);
    }
    return undefined;
  }
}
