import {
  Entity,
  PrimaryGeneratedColumn,
  Column,
  ManyToOne,
  CreateDateColumn,
  JoinColumn,
  OneToMany,
} from "typeorm";
import { ApiProperty } from "@nestjs/swagger";
import { Supplier } from "./supplier.entity";
import { Product } from "./product.entity";
import { SupplierProductRestockLog } from "./supplier-product-restock-log.entity";
import { SupplierProductSize } from "./supplier-product-size.entity";
import { Color } from "./color.entity";

export type SupplierProductStatus = "preorder" | "completed";

@Entity("supplier_products")
export class SupplierProduct {
  @PrimaryGeneratedColumn()
  id: number;

  @Column()
  supplier_id: number;

  @Column()
  product_id: number;

  @Column({ type: "decimal", precision: 10, scale: 2 })
  unit_price: number;

  @Column()
  quantity: number;

  @Column({ name: "color_id", nullable: true })
  color_id: number | null;

  @ManyToOne(() => Color, (color) => color.supplierProducts, {
    nullable: true,
    onDelete: "SET NULL",
  })
  @JoinColumn({ name: "color_id" })
  color: Color | null;

  @Column({
    type: "enum",
    enum: ["preorder", "completed"],
    default: "completed",
  })
  status: SupplierProductStatus;

  @Column({ type: "varchar", length: 50, nullable: true })
  note: string | null;

  @ApiProperty({
    description: "The date when the supplier fulfilled this product purchase",
    type: String,
    format: "date",
    required: false,
    nullable: true,
    example: "2024-03-01",
  })
  @Column({ type: "date", nullable: true })
  purchase_date: Date | null;

  @CreateDateColumn()
  created_at: Date;

  @ManyToOne(() => Supplier, (supplier) => supplier.supplierProducts)
  @JoinColumn({ name: "supplier_id" })
  supplier: Supplier;

  @ManyToOne(() => Product, (product) => product.supplierProducts)
  @JoinColumn({ name: "product_id" })
  product: Product;

  @OneToMany(
    () => SupplierProductRestockLog,
    (restockLog) => restockLog.supplierProduct
  )
  restockLogs: SupplierProductRestockLog[];

  @OneToMany(
    () => SupplierProductSize,
    (supplierProductSize) => supplierProductSize.supplierProduct,
    { cascade: true }
  )
  sizes: SupplierProductSize[];
}
