import {
  ExceptionFilter,
  Catch,
  ArgumentsHost,
  HttpException,
  HttpStatus,
} from "@nestjs/common";
import { Request, Response } from "express";

@Catch(HttpException)
export class HttpExceptionFilter implements ExceptionFilter {
  catch(exception: HttpException, host: ArgumentsHost) {
    const ctx = host.switchToHttp();
    const response = ctx.getResponse<Response>();
    const request = ctx.getRequest<Request>();
    const status = exception.getStatus();

    const errorResponse = {
      statusCode: status,
      timestamp: new Date().toISOString(),
      path: request.url,
      message: exception.message || "Internal server error",
      // NestJS validation errors are often nested, so we extract them
      ...(status === HttpStatus.BAD_REQUEST &&
        (exception.getResponse() as any).message && {
          errors: (exception.getResponse() as any).message,
        }),
    };

    response.status(status).json({
      success: false,
      message: errorResponse.message,
      data: errorResponse,
    });
  }
}

@Catch(Error)
export class AllExceptionsFilter implements ExceptionFilter {
  catch(exception: Error, host: ArgumentsHost) {
    const ctx = host.switchToHttp();
    const response = ctx.getResponse<Response>();
    const request = ctx.getRequest<Request>();

    const status = HttpStatus.INTERNAL_SERVER_ERROR;

    const errorResponse = {
      statusCode: status,
      timestamp: new Date().toISOString(),
      path: request.url,
      message: "Internal server error",
      error: exception.message,
    };

    response.status(status).json({
      success: false,
      message: errorResponse.message,
      data: errorResponse,
    });
  }
}
