import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Patch,
  Post,
  Query,
} from "@nestjs/common";
import {
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiTags,
} from "@nestjs/swagger";
import { OrderService } from "./order.service";
import { CreateOrderDto } from "./dto/create-order.dto";
import { UpdateOrderStatusDto } from "./dto/update-order-status.dto";
import { CreateOrderReturnDto } from "./dto/create-order-return.dto";
import { PaginationDto } from "../common/dto/pagination.dto";

@ApiTags("Orders")
@Controller("orders")
export class OrderController {
  constructor(private readonly orderService: OrderService) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a new sales order" })
  @ApiBody({ type: CreateOrderDto })
  @ApiResponse({ status: 201, description: "Order has been created." })
  @ApiResponse({ status: 400, description: "Validation failed." })
  create(@Body() createOrderDto: CreateOrderDto) {
    return this.orderService.create(createOrderDto);
  }

  @Get()
  @ApiOperation({
    summary: "List orders with pagination and optional search filter",
  })
  @ApiResponse({ status: 200, description: "Returns paginated orders." })
  @ApiQuery({
    name: "limit",
    type: Number,
    required: false,
    description: "Maximum number of orders per page",
  })
  @ApiQuery({
    name: "page",
    type: Number,
    required: false,
    description: "Page number to retrieve",
  })
  @ApiQuery({
    name: "search",
    type: String,
    required: false,
    description:
      "Search keyword applied to order code, status, or customer fields",
  })
  findAll(@Query() paginationDto: PaginationDto) {
    return this.orderService.findAll(paginationDto);
  }

  @Get(":id")
  @ApiOperation({ summary: "Get an order by ID" })
  @ApiParam({ name: "id", description: "Order identifier" })
  @ApiResponse({ status: 200, description: "Returns the requested order." })
  @ApiResponse({ status: 404, description: "Order not found." })
  findOne(@Param("id") id: string) {
    return this.orderService.findOne(+id);
  }

  @Patch(":id/status")
  @ApiOperation({ summary: "Update order status" })
  @ApiParam({ name: "id", description: "Order identifier" })
  @ApiBody({ type: UpdateOrderStatusDto })
  @ApiResponse({ status: 200, description: "Order status updated." })
  updateStatus(
    @Param("id") id: string,
    @Body() updateOrderStatusDto: UpdateOrderStatusDto
  ) {
    return this.orderService.updateStatus(+id, updateOrderStatusDto);
  }

  @Post(":id/returns")
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a return for an order" })
  @ApiParam({ name: "id", description: "Order identifier" })
  @ApiBody({ type: CreateOrderReturnDto })
  @ApiResponse({ status: 201, description: "Order return processed." })
  createReturn(
    @Param("id") id: string,
    @Body() createOrderReturnDto: CreateOrderReturnDto
  ) {
    return this.orderService.createReturn(+id, createOrderReturnDto);
  }

  @Delete(":id")
  @HttpCode(HttpStatus.NO_CONTENT)
  @ApiOperation({ summary: "Delete an order" })
  @ApiParam({ name: "id", description: "Order identifier" })
  @ApiResponse({ status: 204, description: "Order deleted." })
  @ApiResponse({ status: 404, description: "Order not found." })
  remove(@Param("id") id: string) {
    return this.orderService.remove(+id);
  }
}
