import {
  BadRequestException,
  Injectable,
  NotFoundException,
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { DataSource, In, Repository } from "typeorm";
import { Order } from "../entities/order.entity";
import { OrderItem } from "../entities/order-item.entity";
import { OrderReturn } from "../entities/order-return.entity";
import { OrderReturnItem } from "../entities/order-return-item.entity";
import { Customer } from "../entities/customer.entity";
import { Product } from "../entities/product.entity";
import { Size } from "../entities/size.entity";
import { CreateOrderDto } from "./dto/create-order.dto";
import { UpdateOrderStatusDto } from "./dto/update-order-status.dto";
import { CreateOrderReturnDto } from "./dto/create-order-return.dto";
import { StockService } from "../stock/stock.service";
import { PaginationDto } from "../common/dto/pagination.dto";

@Injectable()
export class OrderService {
  constructor(
    @InjectRepository(Order)
    private readonly orderRepository: Repository<Order>,
    private readonly dataSource: DataSource,
    private readonly stockService: StockService
  ) {}

  async findAll(
    paginationDto: PaginationDto,
  ): Promise<{
    data: Order[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(
      Number.parseInt(paginationDto.page ?? "1", 10) || 1,
      1,
    );
    const limit = Math.max(
      Number.parseInt(paginationDto.limit ?? "10", 10) || 10,
      1,
    );
    const search = paginationDto.searchTerm;

    const baseQuery = this.orderRepository
      .createQueryBuilder("orders")
      .leftJoin("orders.customer", "customer");

    if (search) {
      const searchTerm = `%${search.toLowerCase()}%`;
      baseQuery.where(
        "(LOWER(orders.order_code) LIKE :search OR LOWER(orders.status) LIKE :search OR LOWER(customer.first_name) LIKE :search OR LOWER(customer.last_name) LIKE :search OR LOWER(customer.email) LIKE :search OR LOWER(COALESCE(customer.phone, '')) LIKE :search)",
        { search: searchTerm },
      );
    }

    baseQuery.orderBy("orders.created_at", "DESC");

    const total = await baseQuery.clone().getCount();

    const orderIdRows = await baseQuery
      .clone()
      .select("orders.id", "order_id")
      .skip((page - 1) * limit)
      .take(limit)
      .getRawMany();

    const orderIds = orderIdRows
      .map((row) => Number(row.order_id))
      .filter((id) => Number.isInteger(id));

    let data: Order[] = [];
    if (orderIds.length > 0) {
      const orderIndex = new Map(orderIds.map((id, index) => [id, index]));
      const fetchedOrders = await this.orderRepository.find({
        where: { id: In(orderIds) },
        relations: {
          customer: true,
          items: { product: true, size: true },
          returns: { items: true },
        },
        order: { created_at: "DESC" },
      });

      data = fetchedOrders.sort(
        (a, b) =>
          (orderIndex.get(a.id) ?? Number.POSITIVE_INFINITY) -
          (orderIndex.get(b.id) ?? Number.POSITIVE_INFINITY),
      );
    }

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async findOne(id: number): Promise<Order> {
    return this.getOrderByIdOrFail(id);
  }

  async create(createOrderDto: CreateOrderDto): Promise<Order> {
    if (createOrderDto.items.length === 0) {
      throw new BadRequestException("Order must contain at least one item");
    }

    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();

    try {
      const customer = await queryRunner.manager.findOne(Customer, {
        where: { id: createOrderDto.customerId },
      });
      if (!customer) {
        throw new NotFoundException(
          `Customer with ID ${createOrderDto.customerId} not found`
        );
      }

      const order = queryRunner.manager.create(Order, {
        order_code: this.generateOrderCode(),
        customer,
        order_date: createOrderDto.orderDate
          ? new Date(createOrderDto.orderDate)
          : new Date(),
        status: "confirmed",
        total_amount: 0,
        refunded_amount: 0,
      });

      const savedOrder = await queryRunner.manager.save(order);
      let runningTotal = 0;

      for (const itemDto of createOrderDto.items) {
        const product = await queryRunner.manager.findOne(Product, {
          where: { id: itemDto.productId },
        });
        if (!product) {
          throw new NotFoundException(
            `Product with ID ${itemDto.productId} not found`
          );
        }

        const size = await queryRunner.manager.findOne(Size, {
          where: { id: itemDto.sizeId },
        });
        if (!size) {
          throw new NotFoundException(
            `Size with ID ${itemDto.sizeId} not found`
          );
        }

        await this.stockService.decreaseStock({
          productId: product.id,
          sizeId: size.id,
          quantity: itemDto.quantity,
          movementType: "SALE",
          manager: queryRunner.manager,
          referenceType: "ORDER",
          referenceId: savedOrder.id,
        });

        const lineTotal = Number(itemDto.unitPrice) * itemDto.quantity;
        runningTotal += lineTotal;

        const orderItem = queryRunner.manager.create(OrderItem, {
          order: savedOrder,
          product,
          size,
          size_id: size.id,
          quantity: itemDto.quantity,
          returned_quantity: 0,
          unit_price: Number(itemDto.unitPrice),
          line_total: lineTotal,
        });
        await queryRunner.manager.save(orderItem);
      }

      savedOrder.total_amount = runningTotal;
      await queryRunner.manager.save(savedOrder);

      await queryRunner.commitTransaction();
      return this.getOrderByIdOrFail(savedOrder.id);
    } catch (error) {
      await queryRunner.rollbackTransaction();
      throw error;
    } finally {
      await queryRunner.release();
    }
  }

  async updateStatus(
    id: number,
    updateOrderStatusDto: UpdateOrderStatusDto
  ): Promise<Order> {
    const order = await this.getOrderByIdOrFail(id);
    order.status = updateOrderStatusDto.status;
    await this.orderRepository.save(order);
    return this.getOrderByIdOrFail(id);
  }

  async createReturn(
    orderId: number,
    createOrderReturnDto: CreateOrderReturnDto
  ): Promise<Order> {
    if (createOrderReturnDto.items.length === 0) {
      throw new BadRequestException("Return must contain at least one item");
    }

    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();

    try {
      const order = await queryRunner.manager.findOne(Order, {
        where: { id: orderId },
        relations: {
          customer: true,
          items: { product: true, size: true },
          returns: { items: true },
        },
        lock: { mode: "pessimistic_write" },
      });

      if (!order) {
        throw new NotFoundException(`Order with ID ${orderId} not found`);
      }

      const returnEntity = queryRunner.manager.create(OrderReturn, {
        order,
        reason: createOrderReturnDto.reason ?? null,
        total_amount: 0,
      });
      const savedReturn = await queryRunner.manager.save(returnEntity);

      let totalReturnAmount = 0;
      const persistedReturnItems: OrderReturnItem[] = [];

      for (const itemDto of createOrderReturnDto.items) {
        const orderItem = order.items.find((item) => item.id === itemDto.orderItemId);
        if (!orderItem) {
          throw new NotFoundException(
            `Order item with ID ${itemDto.orderItemId} not found in order ${orderId}`
          );
        }

        const remainingQuantity = orderItem.quantity - orderItem.returned_quantity;
        if (itemDto.quantity > remainingQuantity) {
          throw new BadRequestException(
            `Cannot return ${itemDto.quantity} items. Only ${remainingQuantity} remaining for order item ${orderItem.id}`
          );
        }

        orderItem.returned_quantity += itemDto.quantity;
        await queryRunner.manager.save(orderItem);

        const returnAmount = Number(orderItem.unit_price) * itemDto.quantity;
        totalReturnAmount += returnAmount;

        await this.stockService.increaseStock({
          productId: orderItem.product.id,
          sizeId: orderItem.size.id,
          quantity: itemDto.quantity,
          movementType: "RETURN",
          manager: queryRunner.manager,
          referenceType: "ORDER_RETURN",
          referenceId: savedReturn.id,
        });

        const returnItem = queryRunner.manager.create(OrderReturnItem, {
          orderReturn: savedReturn,
          orderItem,
          quantity: itemDto.quantity,
          amount: returnAmount,
        });
        persistedReturnItems.push(await queryRunner.manager.save(returnItem));
      }

      if (persistedReturnItems.length === 0) {
        throw new BadRequestException("No return items were processed");
      }

      savedReturn.total_amount = totalReturnAmount;
      savedReturn.items = persistedReturnItems;
      await queryRunner.manager.save(savedReturn);

      const grossTotal = order.items.reduce(
        (sum, item) => sum + Number(item.line_total),
        0
      );
      const refundedAmount = order.items.reduce(
        (sum, item) => sum + Number(item.unit_price) * item.returned_quantity,
        0
      );

      order.refunded_amount = refundedAmount;
      order.total_amount = grossTotal - refundedAmount;
      await queryRunner.manager.save(order);

      await queryRunner.commitTransaction();
      return this.getOrderByIdOrFail(order.id);
    } catch (error) {
      await queryRunner.rollbackTransaction();
      throw error;
    } finally {
      await queryRunner.release();
    }
  }

  async remove(id: number): Promise<void> {
    const result = await this.orderRepository.delete(id);
    if (result.affected === 0) {
      throw new NotFoundException(`Order with ID ${id} not found`);
    }
  }

  private async getOrderByIdOrFail(id: number): Promise<Order> {
    const order = await this.orderRepository.findOne({
      where: { id },
      relations: {
        customer: true,
        items: { product: true, size: true },
        returns: { items: true },
      },
    });
    if (!order) {
      throw new NotFoundException(`Order with ID ${id} not found`);
    }
    return order;
  }

  private generateOrderCode(): string {
    const now = new Date();
    const datePart = `${now.getFullYear()}${(now.getMonth() + 1)
      .toString()
      .padStart(2, "0")}${now
      .getDate()
      .toString()
      .padStart(2, "0")}`;
    return `ORD-${datePart}-${now.getTime()}`;
  }
}
