import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Patch,
  Post,
  Query,
} from "@nestjs/common";
import {
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiTags,
} from "@nestjs/swagger";
import { PaginationDto } from "../common/dto/pagination.dto";
import { CreateProductDto } from "./dto/create-product.dto";
import { UpdateProductDto } from "./dto/update-product.dto";
import { ProductService } from "./product.service";

@ApiTags("Products")
@Controller("products")
export class ProductController {
  constructor(private readonly productService: ProductService) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a new product" })
  @ApiResponse({
    status: 201,
    description: "The product has been successfully created.",
  })
  @ApiResponse({ status: 400, description: "Bad Request." })
  create(@Body() createProductDto: CreateProductDto) {
    return this.productService.create(createProductDto);
  }

  @Get()
  @ApiOperation({
    summary:
      "Retrieve products with pagination, category relation, and optional search",
  })
  @ApiResponse({ status: 200, description: "Returns paginated products." })
  @ApiQuery({
    name: "limit",
    type: Number,
    required: false,
    description: "Limit of items per page",
  })
  @ApiQuery({
    name: "page",
    type: Number,
    required: false,
    description: "Page number to retrieve",
  })
  @ApiQuery({
    name: "search",
    type: String,
    required: false,
    description:
      "Search keyword applied to product name, code, color, and size fields",
  })
  findAll(@Query() paginationDto: PaginationDto) {
    return this.productService.findAll(paginationDto);
  }

  @Get(":id")
  @ApiOperation({ summary: "Retrieve a product by ID" })
  @ApiParam({ name: "id", description: "ID of the product to retrieve" })
  @ApiResponse({
    status: 200,
    description: "Returns the product with the specified ID.",
  })
  @ApiResponse({ status: 404, description: "Product not found." })
  findOne(@Param("id") id: string) {
    return this.productService.findOne(+id);
  }

  @Patch(":id")
  @ApiOperation({ summary: "Update an existing product" })
  @ApiParam({ name: "id", description: "ID of the product to update" })
  @ApiBody({ type: UpdateProductDto, description: "Product data to update" })
  @ApiResponse({
    status: 200,
    description: "The product has been successfully updated.",
  })
  @ApiResponse({ status: 404, description: "Product not found." })
  @ApiResponse({ status: 400, description: "Bad Request." })
  update(@Param("id") id: string, @Body() updateProductDto: UpdateProductDto) {
    return this.productService.update(+id, updateProductDto);
  }

  @Delete(":id")
  @HttpCode(HttpStatus.NO_CONTENT)
  @ApiOperation({ summary: "Delete a product by ID" })
  @ApiParam({ name: "id", description: "ID of the product to delete" })
  @ApiResponse({
    status: 204,
    description: "The product has been successfully deleted.",
  })
  @ApiResponse({ status: 404, description: "Product not found." })
  remove(@Param("id") id: string) {
    return this.productService.remove(+id);
  }
}
