import { Injectable, NotFoundException, BadRequestException } from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { In, Repository } from "typeorm";
import { Product } from "../entities/product.entity";
import { Category } from "../entities/category.entity";
import { PaginationDto } from "../common/dto/pagination.dto";
import { CreateProductDto } from "./dto/create-product.dto";
import { UpdateProductDto } from "./dto/update-product.dto";
import { Size } from "../entities/size.entity";
import { Color } from "../entities/color.entity";

@Injectable()
export class ProductService {
  constructor(
    @InjectRepository(Product)
    private readonly productRepository: Repository<Product>,
    @InjectRepository(Category)
    private readonly categoryRepository: Repository<Category>,
    @InjectRepository(Size)
    private readonly sizeRepository: Repository<Size>,
    @InjectRepository(Color)
    private readonly colorRepository: Repository<Color>,
  ) {}

  async create(createProductDto: CreateProductDto): Promise<Product> {
    const { categoryId, sizeIds, colorIds, ...productData } =
      createProductDto;

    const category = await this.resolveCategory(categoryId);
    const sizes = await this.resolveSizes(sizeIds);
    const colors = await this.resolveColors(colorIds ?? []);

    const product = this.productRepository.create({
      ...productData,
      category: category ?? null,
      sizes,
      colors,
    });
    return this.productRepository.save(product);
  }

  async findAll(
    paginationDto: PaginationDto,
  ): Promise<{
    data: Product[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(
      Number.parseInt(paginationDto.page ?? "1", 10) || 1,
      1,
    );
    const limit = Math.max(
      Number.parseInt(paginationDto.limit ?? "10", 10) || 10,
      1,
    );
    const search = paginationDto.search?.trim() ?? "";

    const queryBuilder = this.productRepository
      .createQueryBuilder("product")
      .leftJoinAndSelect("product.category", "category")
      .leftJoinAndSelect("product.sizes", "size")
      .leftJoinAndSelect("product.colors", "color")
      .distinct(true);

    if (search) {
      queryBuilder.where(
        "(LOWER(product.product_name) LIKE LOWER(:search) OR LOWER(product.product_code) LIKE LOWER(:search) OR LOWER(product.product_color) LIKE LOWER(:search) OR LOWER(size.name) LIKE LOWER(:search) OR LOWER(color.name) LIKE LOWER(:search))",
        { search: `%${search}%` },
      );
    }

    queryBuilder.orderBy("product.created_at", "DESC");

    const [data, total] = await queryBuilder
      .skip((page - 1) * limit)
      .take(limit)
      .getManyAndCount();

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async findOne(id: number): Promise<Product> {
    const product = await this.productRepository.findOne({
      where: { id },
      relations: ["category", "sizes", "colors"],
    });
    if (!product) {
      throw new NotFoundException(`Product with ID ${id} not found`);
    }
    return product;
  }

  async update(
    id: number,
    updateProductDto: UpdateProductDto,
  ): Promise<Product> {
    const product = await this.findOne(id);
    const { categoryId, sizeIds, colorIds, ...productData } =
      updateProductDto;

    if (categoryId !== undefined) {
      product.category = await this.resolveCategory(categoryId);
    }

    if (sizeIds !== undefined) {
      product.sizes = await this.resolveSizes(sizeIds);
    }

    if (colorIds !== undefined) {
      product.colors = await this.resolveColors(colorIds);
    }

    this.productRepository.merge(product, productData);
    return this.productRepository.save(product);
  }

  async remove(id: number): Promise<void> {
    const result = await this.productRepository.delete(id);
    if (result.affected === 0) {
      throw new NotFoundException(`Product with ID ${id} not found`);
    }
  }

  private async resolveCategory(
    categoryId?: number | null,
  ): Promise<Category | null> {
    if (categoryId === undefined || categoryId === null) {
      return null;
    }

    const category = await this.categoryRepository.findOne({
      where: { id: categoryId },
    });

    if (!category) {
      throw new NotFoundException(`Category with ID ${categoryId} not found`);
    }

    return category;
  }

  private async resolveSizes(sizeIds: number[]): Promise<Size[]> {
    const uniqueIds = Array.from(new Set(sizeIds));
    if (uniqueIds.length === 0) {
      throw new BadRequestException("At least one size must be provided");
    }

    const sizes = await this.sizeRepository.find({
      where: { id: In(uniqueIds) },
    });

    if (sizes.length !== uniqueIds.length) {
      const foundIds = new Set(sizes.map((size) => size.id));
      const missing = uniqueIds.filter((id) => !foundIds.has(id));
      throw new NotFoundException(
        `Sizes not found for identifiers: ${missing.join(", ")}`
      );
    }

    return sizes;
  }

  private async resolveColors(colorIds: number[]): Promise<Color[]> {
    const uniqueIds = Array.from(new Set(colorIds));
    if (uniqueIds.length === 0) {
      return [];
    }

    const colors = await this.colorRepository.find({
      where: { id: In(uniqueIds) },
    });

    if (colors.length !== uniqueIds.length) {
      const foundIds = new Set(colors.map((color) => color.id));
      const missing = uniqueIds.filter((id) => !foundIds.has(id));
      throw new NotFoundException(
        `Colors not found for identifiers: ${missing.join(", ")}`
      );
    }

    return colors;
  }
}
