import { Injectable, NotFoundException } from "@nestjs/common";
import { CreatePurchaseOrderDto } from './dto/create-purchase-order.dto';
import { UpdatePurchaseOrderDto, UpdatePurchaseOrderItemDto } from './dto/update-purchase-order.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { PurchaseOrder } from '../entities/purchase-order.entity';
import { DataSource, Repository, EntityManager } from "typeorm";
import { PurchaseOrderItem } from '../entities/purchase-order-item.entity';
import { Supplier } from '../entities/supplier.entity';
import { Product } from '../entities/product.entity';
import { StockService } from "../stock/stock.service";
import { Size } from "../entities/size.entity";

@Injectable()
export class PurchaseOrderService {
  constructor(
    @InjectRepository(PurchaseOrder)
    private readonly purchaseOrderRepository: Repository<PurchaseOrder>,
    @InjectRepository(PurchaseOrderItem)
    private readonly purchaseOrderItemRepository: Repository<PurchaseOrderItem>,
    @InjectRepository(Supplier)
    private readonly supplierRepository: Repository<Supplier>,
    private readonly dataSource: DataSource,
    private readonly stockService: StockService,
  ) {}

  async createPurchaseOrder(
    createPurchaseOrderDto: CreatePurchaseOrderDto,
  ): Promise<PurchaseOrder> {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();

    try {
      const { supplier_id, order_date, items } = createPurchaseOrderDto;

      const supplier = await queryRunner.manager.findOneBy(Supplier, {
        id: supplier_id,
      });
      if (!supplier) {
        throw new NotFoundException(
          `Supplier with ID ${supplier_id} not found`,
        );
      }

      const purchaseOrder = queryRunner.manager.create(PurchaseOrder, {
        supplier,
        order_date: new Date(order_date),
        total_amount: 0,
      });

      const savedOrder = await queryRunner.manager.save(purchaseOrder);
      let totalOrderAmount = 0;

      for (const itemDto of items) {
        const product = await queryRunner.manager.findOneBy(Product, {
          id: itemDto.product_id,
        });
        if (!product) {
          throw new NotFoundException(
            `Product with ID ${itemDto.product_id} not found`,
          );
        }

        const size = await queryRunner.manager.findOne(Size, {
          where: { id: itemDto.size_id },
        });
        if (!size) {
          throw new NotFoundException(
            `Size with ID ${itemDto.size_id} not found`,
          );
        }

        const itemTotal = itemDto.quantity * itemDto.unit_price;
        totalOrderAmount += itemTotal;

        const orderItem = queryRunner.manager.create(PurchaseOrderItem, {
          purchaseOrder: savedOrder,
          product,
          size,
          size_id: size.id,
          quantity: itemDto.quantity,
          unit_price: itemDto.unit_price,
          total: itemTotal,
        });
        await queryRunner.manager.save(orderItem);

        await this.stockService.increaseStock({
          productId: product.id,
          sizeId: size.id,
          quantity: itemDto.quantity,
          movementType: "NEW",
          manager: queryRunner.manager,
          referenceType: "PURCHASE_ORDER",
          referenceId: savedOrder.id,
        });
      }

      savedOrder.total_amount = totalOrderAmount;
      await queryRunner.manager.save(savedOrder);

      await queryRunner.commitTransaction();
      return this.getPurchaseOrderById(savedOrder.id); // Fetch with relations
    } catch (err) {
      await queryRunner.rollbackTransaction();
      throw err;
    } finally {
      await queryRunner.release();
    }
  }

  async updatePurchaseOrder(
    id: number,
    updatePurchaseOrderDto: UpdatePurchaseOrderDto,
  ): Promise<PurchaseOrder> {
    const queryRunner = this.dataSource.createQueryRunner();
    await queryRunner.connect();
    await queryRunner.startTransaction();

    try {
      let purchaseOrder = await queryRunner.manager.findOne(PurchaseOrder, {
        where: { id },
          relations: [
            "supplier",
            "purchaseOrderItems",
            "purchaseOrderItems.product",
            "purchaseOrderItems.size",
          ],
      });

      if (!purchaseOrder) {
        throw new NotFoundException(`Purchase Order with ID ${id} not found`);
      }

      const originalStockSnapshot = this.buildStockSnapshot(
        purchaseOrder.purchaseOrderItems
      );

      if (updatePurchaseOrderDto.supplier_id) {
        const supplier = await queryRunner.manager.findOneBy(Supplier, {
          id: updatePurchaseOrderDto.supplier_id,
        });
        if (!supplier) {
          throw new NotFoundException(
            `Supplier with ID ${updatePurchaseOrderDto.supplier_id} not found`,
          );
        }
        purchaseOrder.supplier = supplier;
      }

      if (updatePurchaseOrderDto.order_date) {
        purchaseOrder.order_date = new Date(updatePurchaseOrderDto.order_date);
      }

      // Handle items update
      if (updatePurchaseOrderDto.items) {
        let totalOrderAmount = 0;
        const existingItemIds = purchaseOrder.purchaseOrderItems.map(
          (item) => item.id,
        );
        const updatedItemIds = updatePurchaseOrderDto.items
          .filter((item) => (item as UpdatePurchaseOrderItemDto).id)
          .map((item) => (item as UpdatePurchaseOrderItemDto).id);

        // Remove items not present in the update DTO
        const itemsToRemove = existingItemIds.filter(
          (id) => !updatedItemIds.includes(id),
        );
        if (itemsToRemove.length > 0) {
          await queryRunner.manager.delete(PurchaseOrderItem, itemsToRemove);
        }

        for (const itemDto of updatePurchaseOrderDto.items) {
          let orderItem: PurchaseOrderItem | undefined;
          if ((itemDto as UpdatePurchaseOrderItemDto).id) {
            orderItem = purchaseOrder.purchaseOrderItems.find(
              (item) => item.id === (itemDto as UpdatePurchaseOrderItemDto).id,
            );
            if (!orderItem) {
              throw new NotFoundException(
                `Purchase Order Item with ID ${(itemDto as UpdatePurchaseOrderItemDto).id} not found in order ${id}`,
              );
            }
          } else {
            orderItem = queryRunner.manager.create(PurchaseOrderItem, {
              purchaseOrder,
            });
          }

          if (itemDto.product_id) {
            const product = await queryRunner.manager.findOneBy(Product, {
              id: itemDto.product_id,
            });
            if (!product) {
              throw new NotFoundException(
                `Product with ID ${itemDto.product_id} not found`,
              );
            }
            orderItem.product = product;
          }

          if (itemDto.size_id) {
            const size = await queryRunner.manager.findOne(Size, {
              where: { id: itemDto.size_id },
            });
            if (!size) {
              throw new NotFoundException(
                `Size with ID ${itemDto.size_id} not found`
              );
            }
            orderItem.size = size;
            orderItem.size_id = size.id;
          }
          if (itemDto.quantity) orderItem.quantity = itemDto.quantity;
          if (itemDto.unit_price) orderItem.unit_price = itemDto.unit_price;

          // Recalculate item total
          orderItem.total = orderItem.quantity * orderItem.unit_price;
          totalOrderAmount += orderItem.total;
          await queryRunner.manager.save(orderItem);
        }
        purchaseOrder.total_amount = totalOrderAmount;
      } else {
        // If items are not provided in update, recalculate total based on existing items
        purchaseOrder = await queryRunner.manager.findOne(PurchaseOrder, {
          where: { id },
          relations: ["purchaseOrderItems", "purchaseOrderItems.size"],
        });
        if (!purchaseOrder) {
          throw new NotFoundException(`Purchase Order with ID ${id} not found`);
        }
        purchaseOrder.total_amount = purchaseOrder.purchaseOrderItems.reduce(
          (sum, item) => sum + item.total,
          0,
        );
      }

      await queryRunner.manager.save(purchaseOrder);

        const updatedOrderState = await queryRunner.manager.findOne(
          PurchaseOrder,
          {
            where: { id },
            relations: [
              "purchaseOrderItems",
              "purchaseOrderItems.product",
              "purchaseOrderItems.size",
            ],
          }
        );

      if (!updatedOrderState) {
        throw new NotFoundException(
          `Purchase Order with ID ${id} not found after update`
        );
      }

      await this.applyStockDiff(
        originalStockSnapshot,
        this.buildStockSnapshot(updatedOrderState.purchaseOrderItems),
        queryRunner.manager,
        id
      );
      await queryRunner.commitTransaction();
      return this.getPurchaseOrderById(id); // Fetch with relations
    } catch (err) {
      await queryRunner.rollbackTransaction();
      throw err;
    } finally {
      await queryRunner.release();
    }
  }

  async findAllPurchaseOrders(): Promise<PurchaseOrder[]> {
    return this.purchaseOrderRepository.find({
      relations: [
        "supplier",
        "purchaseOrderItems",
        "purchaseOrderItems.product",
        "purchaseOrderItems.size",
      ],
    });
  }

  async getPurchaseOrderById(id: number): Promise<PurchaseOrder> {
    const order = await this.purchaseOrderRepository.findOne({
      where: { id },
      relations: [
        "supplier",
        "purchaseOrderItems",
        "purchaseOrderItems.product",
        "purchaseOrderItems.size",
      ],
    });
    if (!order) {
      throw new NotFoundException(`Purchase Order with ID ${id} not found`);
    }
    return order;
  }

  async findPurchaseOrdersBySupplier(
    supplier_id: number,
  ): Promise<PurchaseOrder[]> {
    const supplier = await this.supplierRepository.findOneBy({
      id: supplier_id,
    });
    if (!supplier) {
      throw new NotFoundException(`Supplier with ID ${supplier_id} not found`);
    }
    return this.purchaseOrderRepository.find({
      where: { supplier: { id: supplier_id } },
      relations: [
        "supplier",
        "purchaseOrderItems",
        "purchaseOrderItems.product",
        "purchaseOrderItems.size",
      ],
    });
  }

  async deletePurchaseOrder(id: number): Promise<void> {
    const result = await this.purchaseOrderRepository.delete(id);
    if (result.affected === 0) {
      throw new NotFoundException(`Purchase Order with ID ${id} not found`);
    }
  }

  private buildStockSnapshot(
    items: PurchaseOrderItem[]
  ): Map<string, { productId: number; sizeId: number; quantity: number }> {
    const snapshot = new Map<
      string,
      { productId: number; sizeId: number; quantity: number }
    >();

    for (const item of items) {
      if (!item.product || !item.size) {
        continue;
      }
      const key = `${item.product.id}:${item.size.id}`;
      const existing = snapshot.get(key);
      if (existing) {
        existing.quantity += item.quantity;
      } else {
        snapshot.set(key, {
          productId: item.product.id,
          sizeId: item.size.id,
          quantity: item.quantity,
        });
      }
    }

    return snapshot;
  }

  private async applyStockDiff(
    originalSnapshot: Map<string, { productId: number; sizeId: number; quantity: number }>,
    updatedSnapshot: Map<string, { productId: number; sizeId: number; quantity: number }>,
    manager: EntityManager,
    referenceId: number
  ): Promise<void> {
    for (const [key, updated] of updatedSnapshot.entries()) {
      const original = originalSnapshot.get(key);
      const originalQuantity = original?.quantity ?? 0;
      const difference = updated.quantity - originalQuantity;

      if (difference > 0) {
        await this.stockService.increaseStock({
          productId: updated.productId,
          sizeId: updated.sizeId,
          quantity: difference,
          movementType: "NEW",
          manager,
          referenceType: "PURCHASE_ORDER_UPDATE",
          referenceId,
        });
      } else if (difference < 0) {
        await this.stockService.decreaseStock({
          productId: updated.productId,
          sizeId: updated.sizeId,
          quantity: Math.abs(difference),
          movementType: "SALE",
          manager,
          referenceType: "PURCHASE_ORDER_UPDATE",
          referenceId,
        });
      }

      if (original) {
        originalSnapshot.delete(key);
      }
    }

    for (const [, original] of originalSnapshot.entries()) {
      await this.stockService.decreaseStock({
        productId: original.productId,
        sizeId: original.sizeId,
        quantity: original.quantity,
        movementType: "SALE",
        manager,
        referenceType: "PURCHASE_ORDER_UPDATE",
        referenceId,
      });
    }
  }
}
