import { DataSource } from "typeorm";
import { dataSourceOptions } from "../database/ormconfig";
import { Supplier } from "../entities/supplier.entity";
import { Product } from "../entities/product.entity";
import { SupplierProduct } from "../entities/supplier-product.entity";
import { PurchaseOrder } from "../entities/purchase-order.entity";
import { PurchaseOrderItem } from "../entities/purchase-order-item.entity";
import { Size } from "../entities/size.entity";
import { Stock } from "../entities/stock.entity";
import { Color } from "../entities/color.entity";

async function seed() {
  const dataSource = new DataSource({
    ...dataSourceOptions,
    logging: false,
  });

  await dataSource.initialize();
  await dataSource.runMigrations();
  console.log("Database connected and migrations applied for seeding.");

  const supplierRepository = dataSource.getRepository(Supplier);
  const productRepository = dataSource.getRepository(Product);
  const supplierProductRepository = dataSource.getRepository(SupplierProduct);
  const purchaseOrderRepository = dataSource.getRepository(PurchaseOrder);
  const purchaseOrderItemRepository = dataSource.getRepository(
    PurchaseOrderItem,
  );
  const sizeRepository = dataSource.getRepository(Size);
  const stockRepository = dataSource.getRepository(Stock);
  const colorRepository = dataSource.getRepository(Color);

  await purchaseOrderItemRepository.delete({});
  await purchaseOrderRepository.delete({});
  await supplierProductRepository.delete({});
  await stockRepository.delete({});
  await productRepository.delete({});
  await supplierRepository.delete({});
  await sizeRepository.delete({});
  await colorRepository.delete({});
  console.log("Cleared existing data.");

  const supplier1 = supplierRepository.create({
    name: "Global Supplies Inc.",
    address: "123 Main St, Anytown",
    email: "contact@globalsupplies.com",
    phone: "555-1111",
  });
  await supplierRepository.save(supplier1);

  const supplier2 = supplierRepository.create({
    name: "Tech Parts Ltd.",
    address: "456 Tech Rd, Techville",
    email: "sales@techparts.com",
    phone: "555-2222",
  });
  await supplierRepository.save(supplier2);
  console.log("Created suppliers.");

  const size15Inch = await sizeRepository.save(
    sizeRepository.create({ name: "15 inch" })
  );
  const sizeStandard = await sizeRepository.save(
    sizeRepository.create({ name: "Standard" })
  );
  const sizeCompact = await sizeRepository.save(
    sizeRepository.create({ name: "Compact" })
  );
  console.log("Created sizes.");

  const colorSpaceGray = await colorRepository.save(
    colorRepository.create({ name: "Space Gray" })
  );
  const colorBlack = await colorRepository.save(
    colorRepository.create({ name: "Black" })
  );
  const colorBlue = await colorRepository.save(
    colorRepository.create({ name: "Blue" })
  );
  console.log("Created colors.");

  const product1 = productRepository.create({
    product_code: "LPX-001",
    product_name: "Laptop Pro X",
    description: "High performance laptop",
    sizes: [size15Inch],
    colors: [colorSpaceGray],
  });
  await productRepository.save(product1);

  const product2 = productRepository.create({
    product_code: "MK-002",
    product_name: "Mechanical Keyboard",
    description: "RGB mechanical keyboard",
    sizes: [sizeStandard],
    colors: [colorBlack],
  });
  await productRepository.save(product2);

  const product3 = productRepository.create({
    product_code: "WM-003",
    product_name: "Wireless Mouse",
    description: "Ergonomic wireless mouse",
    sizes: [sizeCompact],
    colors: [colorBlue],
  });
  await productRepository.save(product3);
  console.log("Created products.");

  const sp1 = supplierProductRepository.create({
    supplier: supplier1,
    product: product1,
    unit_price: 1150.0,
    quantity: 10,
    color: colorSpaceGray,
    color_id: colorSpaceGray.id,
    status: "completed",
    note: "Initial stock allocation",
    purchase_date: new Date("2024-01-10"),
    sizes: [
      {
        size: size15Inch,
        size_id: size15Inch.id,
        quantity: 10,
        color: colorSpaceGray,
        color_id: colorSpaceGray.id,
      },
    ],
  });
  await supplierProductRepository.save(sp1);

  const sp2 = supplierProductRepository.create({
    supplier: supplier1,
    product: product2,
    unit_price: 75.0,
    quantity: 50,
    color: colorBlack,
    color_id: colorBlack.id,
    status: "completed",
    note: "Popular peripheral",
    purchase_date: new Date("2024-02-15"),
    sizes: [
      {
        size: sizeStandard,
        size_id: sizeStandard.id,
        quantity: 50,
        color: colorBlack,
        color_id: colorBlack.id,
      },
    ],
  });
  await supplierProductRepository.save(sp2);

  const sp3 = supplierProductRepository.create({
    supplier: supplier2,
    product: product3,
    unit_price: 30.0,
    quantity: 40,
    color: colorBlue,
    color_id: colorBlue.id,
    status: "preorder",
    note: "Backordered item",
    purchase_date: new Date("2024-03-05"),
    sizes: [
      {
        size: sizeCompact,
        size_id: sizeCompact.id,
        quantity: 40,
        color: colorBlue,
        color_id: colorBlue.id,
      },
    ],
  });
  await supplierProductRepository.save(sp3);
  console.log("Assigned products to suppliers.");

  const purchaseOrder = purchaseOrderRepository.create({
    supplier: supplier1,
    order_date: new Date(),
    total_amount: 0,
  });
  await purchaseOrderRepository.save(purchaseOrder);

  const item1 = purchaseOrderItemRepository.create({
    purchaseOrder,
    product: product1,
    size: size15Inch,
    size_id: size15Inch.id,
    quantity: 5,
    unit_price: 1100.0,
    total: 5 * 1100.0,
  });
  await purchaseOrderItemRepository.save(item1);

  const item2 = purchaseOrderItemRepository.create({
    purchaseOrder,
    product: product2,
    size: sizeStandard,
    size_id: sizeStandard.id,
    quantity: 10,
    unit_price: 70.0,
    total: 10 * 70.0,
  });
  await purchaseOrderItemRepository.save(item2);

  purchaseOrder.total_amount = item1.total + item2.total;
  await purchaseOrderRepository.save(purchaseOrder);
  console.log("Created sample purchase order.");

  await dataSource.destroy();
  console.log("Seeding complete.");
}

seed().catch((error) => console.error("Seeding failed:", error));
