import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Patch,
  Post,
  Query,
} from "@nestjs/common";
import {
  ApiBody,
  ApiOperation,
  ApiParam,
  ApiQuery,
  ApiResponse,
  ApiTags,
} from "@nestjs/swagger";
import { PaginationDto } from "../common/dto/pagination.dto";
import { CreateSizeDto } from "./dto/create-size.dto";
import { UpdateSizeDto } from "./dto/update-size.dto";
import { SizeService } from "./size.service";

@ApiTags("Sizes")
@Controller("sizes")
export class SizeController {
  constructor(private readonly sizeService: SizeService) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a new size" })
  @ApiResponse({
    status: 201,
    description: "The size has been successfully created.",
  })
  @ApiResponse({ status: 400, description: "Validation failed." })
  create(@Body() createSizeDto: CreateSizeDto) {
    return this.sizeService.create(createSizeDto);
  }

  @Get()
  @ApiOperation({
    summary: "Retrieve sizes with pagination and optional search",
  })
  @ApiResponse({
    status: 200,
    description: "Returns sizes with pagination metadata.",
  })
  @ApiResponse({ status: 400, description: "Validation failed." })
  @ApiQuery({
    name: "limit",
    type: Number,
    required: false,
    description: "Limit of items per page",
  })
  @ApiQuery({
    name: "page",
    type: Number,
    required: false,
    description: "Page of items",
  })
  @ApiQuery({
    name: "query",
    type: String,
    required: false,
    description: "Search keyword (alias: search)",
  })
  @ApiQuery({
    name: "search",
    type: String,
    required: false,
    description: "Deprecated search keyword; prefer query",
  })
  findAll(@Query() paginationDto: PaginationDto) {
    return this.sizeService.findAll(paginationDto);
  }

  @Get(":id")
  @ApiOperation({ summary: "Retrieve a size by ID" })
  @ApiParam({ name: "id", description: "Size identifier" })
  @ApiResponse({
    status: 200,
    description: "Returns the size with the specified ID.",
  })
  @ApiResponse({ status: 404, description: "Size not found." })
  findOne(@Param("id") id: string) {
    return this.sizeService.findOne(+id);
  }

  @Patch(":id")
  @ApiOperation({ summary: "Update an existing size" })
  @ApiParam({ name: "id", description: "Size identifier" })
  @ApiBody({ type: UpdateSizeDto })
  @ApiResponse({
    status: 200,
    description: "The size has been successfully updated.",
  })
  @ApiResponse({ status: 404, description: "Size not found." })
  update(@Param("id") id: string, @Body() updateSizeDto: UpdateSizeDto) {
    return this.sizeService.update(+id, updateSizeDto);
  }

  @Delete(":id")
  @HttpCode(HttpStatus.NO_CONTENT)
  @ApiOperation({ summary: "Delete a size" })
  @ApiParam({ name: "id", description: "Size identifier" })
  @ApiResponse({ status: 204, description: "Size removed." })
  @ApiResponse({ status: 404, description: "Size not found." })
  async remove(@Param("id") id: string) {
    await this.sizeService.remove(+id);
  }
}

