import {
  BadRequestException,
  Injectable,
  NotFoundException,
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { QueryFailedError, Repository } from "typeorm";
import { PaginationDto } from "../common/dto/pagination.dto";
import { Size } from "../entities/size.entity";
import { CreateSizeDto } from "./dto/create-size.dto";
import { UpdateSizeDto } from "./dto/update-size.dto";

@Injectable()
export class SizeService {
  constructor(
    @InjectRepository(Size)
    private readonly sizeRepository: Repository<Size>,
  ) {}

  async create(createSizeDto: CreateSizeDto): Promise<Size> {
    const size = this.sizeRepository.create(createSizeDto);
    try {
      return await this.sizeRepository.save(size);
    } catch (error) {
      this.handlePersistenceError(error);
    }
  }

  async findAll(
    paginationDto: PaginationDto,
  ): Promise<{
    data: Size[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(
      Number.parseInt(paginationDto.page ?? "1", 10) || 1,
      1,
    );
    const limit = Math.max(
      Number.parseInt(paginationDto.limit ?? "10", 10) || 10,
      1,
    );
    const search = paginationDto.searchTerm;

    const queryBuilder = this.sizeRepository
      .createQueryBuilder("size")
      .orderBy("size.name", "ASC");

    if (search) {
      queryBuilder.where("LOWER(size.name) LIKE LOWER(:search)", {
        search: `%${search}%`,
      });
    }

    const [data, total] = await queryBuilder
      .skip((page - 1) * limit)
      .take(limit)
      .getManyAndCount();

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async findOne(id: number): Promise<Size> {
    const size = await this.sizeRepository.findOne({ where: { id } });
    if (!size) {
      throw new NotFoundException(`Size with ID ${id} not found`);
    }
    return size;
  }

  async update(id: number, updateSizeDto: UpdateSizeDto): Promise<Size> {
    const size = await this.findOne(id);
    this.sizeRepository.merge(size, updateSizeDto);
    try {
      return await this.sizeRepository.save(size);
    } catch (error) {
      this.handlePersistenceError(error);
    }
  }

  async remove(id: number): Promise<void> {
    const size = await this.findOne(id);
    await this.sizeRepository.remove(size);
  }

  private handlePersistenceError(error: unknown): never {
    if (error instanceof QueryFailedError) {
      const driverError = (error as QueryFailedError & { code?: string }).code;
      if (driverError === "23505" || driverError === "ER_DUP_ENTRY") {
        throw new BadRequestException("Size name must be unique");
      }
    }
    throw error;
  }
}

