import {
  Body,
  Controller,
  Get,
  HttpCode,
  HttpStatus,
  Post,
  Query,
} from "@nestjs/common";
import {
  ApiBody,
  ApiOperation,
  ApiQuery,
  ApiResponse,
  ApiTags,
} from "@nestjs/swagger";
import { StockService } from "./stock.service";
import { CreateStockDto } from "./dto/create-stock.dto";
import { StockReportQueryDto } from "./dto/stock-report-query.dto";

@ApiTags("Stock")
@Controller("stocks")
export class StockController {
  constructor(private readonly stockService: StockService) {}

  @Get()
  @ApiOperation({ summary: "List all stock levels" })
  @ApiResponse({ status: 200, description: "Current stock quantities" })
  findAll() {
    return this.stockService.listStocks();
  }

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Add new stock quantity for a product" })
  @ApiBody({ type: CreateStockDto })
  @ApiResponse({
    status: 201,
    description: "Stock quantity has been updated.",
  })
  addStock(@Body() createStockDto: CreateStockDto) {
    return this.stockService.addNewStock(createStockDto);
  }

  @Get("report")
  @ApiOperation({ summary: "Generate a monthly stock report" })
  @ApiQuery({
    name: "month",
    required: false,
    description: "Target month in YYYY-MM format",
    example: "2025-07",
  })
  @ApiResponse({ status: 200, description: "AntD-compatible stock report" })
  getReport(@Query("month") month?: string) {
    return this.stockService.generateMonthlyReport(month);
  }

  @Get("report/supplier-product")
  @ApiOperation({
    summary: "Generate product-wise supplier stock report",
    description:
      "Aggregates stock movements by product for the provided date range and returns size/color level breakdown for new stock.",
  })
  @ApiQuery({
    name: "startDate",
    required: true,
    description: "Start date (inclusive) in YYYY-MM-DD format",
  })
  @ApiQuery({
    name: "endDate",
    required: true,
    description: "End date (inclusive) in YYYY-MM-DD format",
  })
  @ApiResponse({
    status: 200,
    description: "Product-wise supplier stock report",
  })
  getSupplierProductReport(@Query() query: StockReportQueryDto) {
    return this.stockService.generateSupplierProductReport(query);
  }

  @Get("report/variants/daily")
  @ApiOperation({
    summary: "Generate day-wise variant report for a date range",
    description:
      "Returns one row per day within the provided range, listing purchases with size/color details and cost rollups.",
  })
  @ApiQuery({
    name: "startDate",
    required: true,
    description: "Start date (inclusive) in YYYY-MM-DD format",
  })
  @ApiQuery({
    name: "endDate",
    required: true,
    description: "End date (inclusive) in YYYY-MM-DD format",
  })
  @ApiResponse({
    status: 200,
    description: "Day-wise stock intake breakdown",
  })
  @Get("report/status")
  @ApiOperation({
    summary: "Generate supplier status stock report",
    description:
      "Summarises stock quantities by product, size, color and supplier status (preorder/completed).",
  })
  @ApiResponse({
    status: 200,
    description: "Supplier status stock breakdown",
  })
  getStatusReport() {
    return this.stockService.generateStatusReport();
  }
}
