import {
  BadRequestException,
  Injectable,
  NotFoundException,
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import {
  DataSource,
  EntityManager,
  FindOptionsWhere,
  IsNull,
  Repository,
} from "typeorm";
import { Color } from "../entities/color.entity";
import { Product } from "../entities/product.entity";
import { Size } from "../entities/size.entity";
import {
  StockMovement,
  StockMovementType,
} from "../entities/stock-movement.entity";
import { Stock } from "../entities/stock.entity";
import { SupplierProduct } from "../entities/supplier-product.entity";
import { CreateStockDto } from "./dto/create-stock.dto";
import { StockReportQueryDto } from "./dto/stock-report-query.dto";

interface AdjustStockInput {
  productId: number;
  sizeId: number;
  quantity: number;
  movementType: StockMovementType;
  manager?: EntityManager;
  referenceType?: string;
  referenceId?: number;
  createIfMissing?: boolean;
  colorId?: number | null;
}

@Injectable()
export class StockService {
  constructor(
    @InjectRepository(Stock)
    private readonly stockRepository: Repository<Stock>,
    @InjectRepository(StockMovement)
    private readonly stockMovementRepository: Repository<StockMovement>,
    @InjectRepository(SupplierProduct)
    private readonly supplierProductRepository: Repository<SupplierProduct>,
    private readonly dataSource: DataSource
  ) {}

  listStocks(): Promise<Stock[]> {
    return this.stockRepository.find({
      relations: { product: true, size: true, color: true },
      order: { updated_at: "DESC" },
    });
  }

  async addNewStock(createStockDto: CreateStockDto): Promise<Stock> {
    return this.increaseStock({
      productId: createStockDto.productId,
      sizeId: createStockDto.sizeId,
      quantity: createStockDto.quantity,
      colorId: createStockDto.colorId ?? null,
      movementType: "NEW",
      referenceType: createStockDto.referenceType ?? "MANUAL",
      referenceId: createStockDto.referenceId,
      createIfMissing: true,
    });
  }

  async increaseStock(input: AdjustStockInput): Promise<Stock> {
    if (input.quantity <= 0) {
      throw new BadRequestException("Quantity must be greater than zero");
    }
    if (input.manager) {
      return this.adjustStock({
        ...input,
        manager: input.manager,
        increase: true,
      });
    }

    return this.dataSource.transaction((manager) =>
      this.adjustStock({
        ...input,
        manager,
        increase: true,
      })
    );
  }

  async decreaseStock(input: AdjustStockInput): Promise<Stock> {
    if (input.quantity <= 0) {
      throw new BadRequestException("Quantity must be greater than zero");
    }
    if (input.manager) {
      return this.adjustStock({
        ...input,
        manager: input.manager,
        increase: false,
      });
    }

    return this.dataSource.transaction((manager) =>
      this.adjustStock({
        ...input,
        manager,
        increase: false,
      })
    );
  }

  private async adjustStock(
    input: AdjustStockInput & { increase: boolean; manager: EntityManager }
  ): Promise<Stock> {
    const manager = input.manager;
    const hasColorFilter = input.colorId !== undefined;
    const targetColorId = hasColorFilter ? (input.colorId ?? null) : undefined;

    const product = await manager.findOne(Product, {
      where: { id: input.productId },
    });
    if (!product) {
      throw new NotFoundException(
        `Product with ID ${input.productId} not found`
      );
    }

    const size = await manager.findOne(Size, {
      where: { id: input.sizeId },
    });
    if (!size) {
      throw new NotFoundException(`Size with ID ${input.sizeId} not found`);
    }

    let color: Color | null = null;
    if (targetColorId !== undefined && targetColorId !== null) {
      color = await manager.findOne(Color, {
        where: { id: targetColorId },
      });
      if (!color) {
        throw new NotFoundException(`Color with ID ${targetColorId} not found`);
      }
    }

    const whereClause: FindOptionsWhere<Stock> = {
      product: { id: product.id },
      size: { id: size.id },
    };
    if (hasColorFilter) {
      if (targetColorId === null) {
        whereClause.color_id = IsNull();
      } else if (targetColorId !== undefined) {
        whereClause.color_id = targetColorId;
      }
    }

    let stock = await manager.findOne(Stock, {
      where: whereClause,
      lock: { mode: "pessimistic_write" },
      relations: { product: true, size: true, color: true },
    });
    if (!stock) {
      if (!input.increase && !input.createIfMissing) {
        throw new NotFoundException(
          `Stock not found for product ${product.product_name} (size ${size.name})`
        );
      }

      stock = manager.create(Stock, {
        product,
        size,
        quantity: 0,
        color: hasColorFilter ? (color ?? null) : null,
        color_id:
          hasColorFilter && targetColorId !== undefined
            ? (color?.id ?? null)
            : null,
      });
    }

    if (input.increase) {
      stock.quantity += input.quantity;
    } else {
      if (stock.quantity < input.quantity) {
        throw new BadRequestException(
          `Insufficient stock for product ${product.product_name} (size ${size.name}). Available: ${stock.quantity}, requested: ${input.quantity}`
        );
      }
      stock.quantity -= input.quantity;
    }

    if (input.colorId !== undefined) {
      stock.color = color ?? null;
      stock.color_id = color?.id ?? null;
    }

    const savedStock = await manager.save(Stock, stock);
    const movement = manager.create(StockMovement, {
      stock: savedStock,
      movement_type: input.movementType,
      quantity: input.quantity,
      reference_type: input.referenceType ?? null,
      reference_id: input.referenceId ?? null,
    });
    await manager.save(StockMovement, movement);

    return savedStock;
  }

  async generateSupplierProductReport(query: StockReportQueryDto) {
    const { startDate, endDate } = query;
    if (!startDate || !endDate) {
      throw new BadRequestException(
        "startDate and endDate query params are required"
      );
    }

    const { start, endExclusive } = this.resolveDateRange(startDate, endDate);

    const movements = await this.stockMovementRepository
      .createQueryBuilder("movement")
      .innerJoin("movement.stock", "stock")
      .innerJoin("stock.product", "product")
      .leftJoin("stock.size", "size")
      .leftJoin("stock.color", "color")
      .leftJoin(
        SupplierProduct,
        "supplierProduct",
        "supplierProduct.id = movement.reference_id AND movement.reference_type = 'SUPPLIER_PRODUCT'"
      )
      .select([
        "movement.stock_id AS stockId",
        "movement.movement_type AS movementType",
        "movement.quantity AS quantity",
        "movement.created_at AS createdAt",
        "product.id AS productId",
        "product.product_name AS productName",
        "product.product_code AS productCode",
        "size.id AS sizeId",
        "size.name AS sizeName",
        "color.id AS colorId",
        "color.name AS colorName",
        "supplierProduct.status AS supplierProductStatus",
      ])
      .where("movement.created_at >= :start AND movement.created_at < :end", {
        start,
        end: endExclusive,
      })
      .orderBy("product.product_name", "ASC")
      .addOrderBy("movement.created_at", "ASC")
      .getRawMany();

    if (movements.length === 0) {
      return {
        meta: {
          generatedAt: new Date().toISOString(),
          startDate,
          endDate,
          startInclusive: start.toISOString(),
          endExclusive: endExclusive.toISOString(),
        },
        data: [],
      };
    }

    const productEntries = new Map<
      number,
      {
        productId: number;
        productName: string;
        productCode: string | null;
        dates: Map<
          string,
          {
            firstStockId: number | null;
            todayNewStock: number;
            bookedOrder: number;
            completedStock: number;
            returns: number;
            sizeVariants: Map<
              string,
              {
                size_id: number | null;
                size_name: string | null;
                color: string | null;
                stock: number;
              }
            >;
          }
        >;
      }
    >();

    const productIds: number[] = [];

    for (const row of movements) {
      const productId = Number(row.productId);
      if (!productId) {
        continue;
      }
      if (!productEntries.has(productId)) {
        productEntries.set(productId, {
          productId,
          productName: row.productName ?? "Unknown Product",
          productCode: row.productCode ?? null,
          dates: new Map(),
        });
        productIds.push(productId);
      }

      const entry = productEntries.get(productId)!;
      const dateKey = toDateKey(row.createdAt);
      let metrics = entry.dates.get(dateKey);
      if (!metrics) {
        metrics = {
          firstStockId: null,
          todayNewStock: 0,
          bookedOrder: 0,
          completedStock: 0,
          returns: 0,
          sizeVariants: new Map(),
        };
        entry.dates.set(dateKey, metrics);
      }

      const quantity = Number(row.quantity ?? 0);
      if (!quantity) {
        continue;
      }

      const parsedStockId =
        row.stockId !== null && row.stockId !== undefined
          ? Number(row.stockId)
          : null;
      if (metrics.firstStockId === null && parsedStockId) {
        metrics.firstStockId = parsedStockId;
      }

      const movementType = row.movementType as StockMovementType;

      if (movementType === "NEW") {
        metrics.todayNewStock += quantity;
        if ((row.supplierProductStatus ?? null) === "completed") {
          metrics.completedStock += quantity;
        }

        const variantKey = `${row.sizeId ?? "null"}::${row.colorId ?? "null"}`;
        let variant = metrics.sizeVariants.get(variantKey);
        if (!variant) {
          const sizeId =
            row.sizeId !== null && row.sizeId !== undefined
              ? Number(row.sizeId)
              : null;
          variant = {
            size_id: sizeId,
            size_name: row.sizeName ?? null,
            color: row.colorName ?? null,
            stock: 0,
          };
        }
        variant.stock += quantity;
        metrics.sizeVariants.set(variantKey, variant);
      } else if (movementType === "SALE") {
        metrics.bookedOrder += quantity;
      } else if (movementType === "RETURN") {
        metrics.returns += quantity;
      }
    }

    const openingRows = await this.stockMovementRepository
      .createQueryBuilder("movement")
      .innerJoin("movement.stock", "stock")
      .innerJoin("stock.product", "product")
      .select("product.id", "productId")
      .addSelect(
        "SUM(CASE WHEN movement.movement_type IN ('NEW','RETURN') THEN movement.quantity ELSE -movement.quantity END)",
        "quantity"
      )
      .where("product.id IN (:...productIds)", { productIds })
      .andWhere("movement.created_at < :start", { start })
      .groupBy("product.id")
      .getRawMany();

    const openingMap = new Map<number, number>();
    for (const row of openingRows) {
      const productId = Number(row.productId);
      const quantity = Number(row.quantity ?? 0);
      openingMap.set(productId, quantity);
    }

    const sortedProducts = Array.from(productEntries.values()).sort((a, b) => {
      const nameCompare = (a.productName ?? "").localeCompare(
        b.productName ?? ""
      );
      if (nameCompare !== 0) {
        return nameCompare;
      }
      return a.productId - b.productId;
    });

    const data: Array<{
      stockId: number | null;
      productId: number;
      date: string;
      productName: string;
      todayNewStock: number;
      bookedOrder: number;
      completedStock: number;
      productCode: string | null;
      return: number;
      openingStock: number;
      closing: number;
      sizeWithColor: Array<{
        size_id: number | null;
        size_name: string | null;
        color: string | null;
        stock: number;
      }>;
    }> = [];

    for (const product of sortedProducts) {
      const dateKeys = Array.from(product.dates.keys()).sort();
      let opening = openingMap.get(product.productId) ?? 0;

      for (const dateKey of dateKeys) {
        const metrics = product.dates.get(dateKey)!;
        const closing =
          opening +
          metrics.todayNewStock +
          metrics.returns -
          metrics.bookedOrder;
        const sizeWithColor = Array.from(metrics.sizeVariants.values()).sort(
          (a, b) => {
            const sizeCompare = (a.size_name ?? "").localeCompare(
              b.size_name ?? ""
            );
            if (sizeCompare !== 0) {
              return sizeCompare;
            }
            return (a.color ?? "").localeCompare(b.color ?? "");
          }
        );

        data.push({
          stockId: metrics.firstStockId,
          productId: product.productId,
          date: dateKey,
          productName: product.productName,
          todayNewStock: metrics.todayNewStock,
          bookedOrder: metrics.bookedOrder,
          completedStock: metrics.completedStock,
          productCode: product.productCode,
          return: metrics.returns,
          openingStock: opening,
          closing,
          sizeWithColor,
        });

        opening = closing;
      }
    }

    return {
      meta: {
        generatedAt: new Date().toISOString(),
        startDate,
        endDate,
        startInclusive: start.toISOString(),
        endExclusive: endExclusive.toISOString(),
      },
      data,
    };
  }

  async generateStatusReport() {
    const [supplierProducts, stocks] = await Promise.all([
      this.supplierProductRepository.find({
        relations: {
          product: true,
          sizes: { size: true, color: true },
        },
      }),
      this.stockRepository.find({
        relations: { product: true, size: true, color: true },
      }),
    ]);

    const stockMap = new Map<string, number>();
    for (const stock of stocks) {
      const key = this.buildVariantKey(
        stock.product?.id ?? 0,
        stock.size?.id ?? 0,
        stock.color?.id ?? null
      );
      stockMap.set(key, (stockMap.get(key) ?? 0) + stock.quantity);
    }

    const dataMap = new Map<
      string,
      {
        productId: number | null;
        productName: string | null;
        productCode: string | null;
        sizeId: number | null;
        size: string | null;
        colorId: number | null;
        color: string | null;
        preorderQuantity: number;
        completedQuantity: number;
        currentStock: number;
        orderQuantity: number;
      }
    >();

    for (const supplierProduct of supplierProducts) {
      const product = supplierProduct.product;
      for (const sizeEntry of supplierProduct.sizes ?? []) {
        const size = sizeEntry.size;
        const color = sizeEntry.color;
        const key = this.buildVariantKey(
          product?.id ?? 0,
          size?.id ?? 0,
          color?.id ?? null
        );
        let row = dataMap.get(key);
        if (!row) {
          row = {
            productId: product?.id ?? null,
            productName: product?.product_name ?? null,
            productCode: product?.product_code ?? null,
            sizeId: size?.id ?? null,
            size: size?.name ?? null,
            colorId: color?.id ?? null,
            color: color?.name ?? null,
            preorderQuantity: 0,
            completedQuantity: 0,
            currentStock: stockMap.get(key) ?? 0,
            orderQuantity: 0,
          };
          dataMap.set(key, row);
        }
        const targetField =
          supplierProduct.status === "preorder"
            ? "preorderQuantity"
            : "completedQuantity";
        row[targetField] += sizeEntry.quantity;
        row.orderQuantity = row.preorderQuantity + row.completedQuantity;
      }
    }

    const report = Array.from(dataMap.values()).sort((a, b) => {
      const nameCompare = (a.productName ?? "").localeCompare(
        b.productName ?? ""
      );
      if (nameCompare !== 0) {
        return nameCompare;
      }
      const sizeCompare = (a.size ?? "").localeCompare(b.size ?? "");
      if (sizeCompare !== 0) {
        return sizeCompare;
      }
      return (a.color ?? "").localeCompare(b.color ?? "");
    });

    const totals = report.reduce(
      (acc, row) => {
        acc.preorder += row.preorderQuantity;
        acc.completed += row.completedQuantity;
        acc.order += row.orderQuantity;
        acc.stock += row.currentStock;
        return acc;
      },
      { preorder: 0, completed: 0, order: 0, stock: 0 }
    );

    return {
      meta: {
        generatedAt: new Date().toISOString(),
        totals,
      },
      data: report,
    };
  }

  private buildVariantKey(
    productId: number,
    sizeId: number,
    colorId: number | null
  ): string {
    return `${productId}::${sizeId}::${colorId ?? "null"}`;
  }

  private resolveDateRange(
    startDate: string,
    endDate: string
  ): {
    start: Date;
    endExclusive: Date;
  } {
    const start = this.normalizeDateOnly(startDate, "startDate");
    const inclusiveEnd = this.normalizeDateOnly(endDate, "endDate");

    if (inclusiveEnd.getTime() < start.getTime()) {
      throw new BadRequestException("endDate must be on or after startDate");
    }

    const endExclusive = new Date(inclusiveEnd.getTime() + 24 * 60 * 60 * 1000);
    return { start, endExclusive };
  }

  private normalizeDateOnly(input: string, fieldName: string): Date {
    const match = /^(\d{4})-(\d{2})-(\d{2})$/.exec(input);
    if (!match) {
      throw new BadRequestException(
        `${fieldName} must be provided in YYYY-MM-DD format`
      );
    }
    const [, yearStr, monthStr, dayStr] = match;
    const year = Number(yearStr);
    const monthIndex = Number(monthStr) - 1;
    const day = Number(dayStr);
    const date = new Date(Date.UTC(year, monthIndex, day));
    if (Number.isNaN(date.getTime())) {
      throw new BadRequestException(
        `${fieldName} must be provided in YYYY-MM-DD format`
      );
    }
    return date;
  }

  async generateMonthlyReport(
    month?: string
  ): Promise<Record<string, unknown>> {
    const { start, end, daysInMonth, label, metaMonth } =
      this.resolveMonth(month);

    const stocks = await this.stockRepository.find({
      relations: { product: true, size: true, color: true },
    });

    stocks.sort((a, b) => {
      const nameA = a.product?.product_name ?? "";
      const nameB = b.product?.product_name ?? "";
      const nameCompare = nameA.localeCompare(nameB);
      if (nameCompare !== 0) {
        return nameCompare;
      }
      const sizeCompare = (a.size?.name ?? "").localeCompare(
        b.size?.name ?? ""
      );
      if (sizeCompare !== 0) {
        return sizeCompare;
      }
      return (a.color?.name ?? "").localeCompare(b.color?.name ?? "");
    });

    const columns = this.buildReportColumns();

    if (stocks.length === 0) {
      return {
        title: `Stock Report - ${label}`,
        rowKey: "key",
        meta: {
          month: metaMonth,
          daysInMonth,
        },
        columns,
        dataSource: [
          {
            key: "summary",
            isSummary: true,
            item: "Total",
            opening: 0,
            closing: 0,
          },
        ],
      };
    }

    const stockIds = stocks.map((stock) => stock.id);

    const openingRows = await this.stockMovementRepository
      .createQueryBuilder("movement")
      .select("movement.stock_id", "stockId")
      .addSelect(
        "SUM(CASE WHEN movement.movement_type IN ('NEW','RETURN') THEN movement.quantity ELSE -movement.quantity END)",
        "quantity"
      )
      .where("movement.stock_id IN (:...stockIds)", { stockIds })
      .andWhere("movement.created_at < :start", { start })
      .groupBy("movement.stock_id")
      .getRawMany();

    const openingMap = new Map<number, number>();
    for (const row of openingRows) {
      openingMap.set(Number(row.stockId), Number(row.quantity ?? 0));
    }

    const monthlyMovements = await this.stockMovementRepository
      .createQueryBuilder("movement")
      .select([
        "movement.stock_id AS stockId",
        "movement.movement_type AS movementType",
        "movement.quantity AS quantity",
        "movement.created_at AS createdAt",
      ])
      .where("movement.stock_id IN (:...stockIds)", { stockIds })
      .andWhere(
        "movement.created_at >= :start AND movement.created_at < :end",
        {
          start,
          end,
        }
      )
      .orderBy("movement.created_at", "ASC")
      .getRawMany();

    const movementMap = new Map<
      number,
      Array<{ day: number; type: StockMovementType; quantity: number }>
    >();

    for (const row of monthlyMovements) {
      const stockId = Number(row.stockId);
      const quantity = Number(row.quantity);
      if (!stockId || quantity === 0) {
        continue;
      }
      const movementType = row.movementType as StockMovementType;
      const movementDate = new Date(row.createdAt);
      const day = movementDate.getUTCDate();
      if (day < 1 || day > 31) {
        continue;
      }
      const existing = movementMap.get(stockId) ?? [];
      existing.push({ day, type: movementType, quantity });
      movementMap.set(stockId, existing);
    }

    const rows = [] as Array<Record<string, unknown>>;
    let totalOpening = 0;
    let totalClosing = 0;

    const allDayKeys = Array.from({ length: 31 }, (_, idx) => `${idx + 1}`);

    stocks.forEach((stock, index) => {
      const opening = Number(openingMap.get(stock.id) ?? 0);

      const rawDays: Record<string, { n: number; r: number; s: number }> = {};
      const displayDays: Record<string, string> = {};

      for (const dayKey of allDayKeys) {
        rawDays[dayKey] = { n: 0, r: 0, s: 0 };
        displayDays[dayKey] = "";
      }

      const movements = movementMap.get(stock.id) ?? [];

      let totalNew = 0;
      let totalReturns = 0;
      let totalSold = 0;

      for (const movement of movements) {
        const key = `${movement.day}`;
        const entry = rawDays[key];
        if (!entry) {
          continue;
        }
        if (movement.type === "NEW") {
          entry.n += movement.quantity;
        } else if (movement.type === "RETURN") {
          entry.r += movement.quantity;
        } else if (movement.type === "SALE") {
          entry.s += movement.quantity;
        }
      }

      for (let day = 1; day <= 31; day += 1) {
        const key = `${day}`;
        const entry = rawDays[key];
        const tokens: string[] = [];
        if (entry.n > 0) {
          tokens.push("N");
        }
        if (entry.r > 0) {
          tokens.push("R");
        }
        if (entry.s > 0) {
          tokens.push(String(entry.s));
        }
        displayDays[key] = tokens.join("/");

        if (day <= daysInMonth) {
          totalNew += entry.n;
          totalReturns += entry.r;
          totalSold += entry.s;
        }
      }

      const closing = opening + totalNew + totalReturns - totalSold;

      totalOpening += opening;
      totalClosing += closing;

      const product = stock.product;
      const sizeLabel = stock.size?.name ?? "N/A";
      const colorLabel = stock.color?.name ?? "N/A";

      const imageUrl = product?.product_code
        ? `https://dummyimage.com/120x120/0f172a/f8fafc&text=${encodeURIComponent(
            product.product_code
          )}`
        : "https://dummyimage.com/120x120/0f172a/f8fafc&text=Shoe";

      rows.push({
        key: `stock-${stock.id}`,
        sl: index + 1,
        item: product
          ? `${product.product_name} / ${product.product_code}`
          : "Unknown Product",
        image: imageUrl,
        size: sizeLabel,
        color: colorLabel,
        opening,
        days: {
          raw: rawDays,
          display: displayDays,
        },
        closing,
      });
    });

    rows.push({
      key: "summary",
      isSummary: true,
      item: "Total",
      opening: totalOpening,
      closing: totalClosing,
    });

    return {
      title: `Stock Report - ${label}`,
      rowKey: "key",
      meta: {
        month: metaMonth,
        daysInMonth,
      },
      columns,
      dataSource: rows,
    };
  }

  private resolveReportRange(
    startDate?: string,
    endDate?: string
  ): {
    start: Date | undefined;
    end: Date | undefined;
    displayStart: string | undefined;
    displayEnd: string | undefined;
  } {
    let start: Date | undefined;
    let endExclusive: Date | undefined;

    if (startDate) {
      start = this.parseYmdDate(startDate);
    }

    if (endDate) {
      const endInclusive = this.parseYmdDate(endDate);
      endExclusive = new Date(endInclusive.getTime() + 24 * 60 * 60 * 1000);
    }

    if (start && endExclusive && start >= endExclusive) {
      throw new BadRequestException(
        "endDate must be later than startDate when both are provided"
      );
    }

    return {
      start,
      end: endExclusive,
      displayStart: startDate,
      displayEnd: endDate,
    };
  }

  private parseYmdDate(value: string): Date {
    const matches = /^(\d{4})-(\d{2})-(\d{2})$/.exec(value);
    if (!matches) {
      throw new BadRequestException(
        "Dates must be provided in YYYY-MM-DD format"
      );
    }
    const year = Number(matches[1]);
    const monthIndex = Number(matches[2]) - 1;
    const day = Number(matches[3]);

    const date = new Date(Date.UTC(year, monthIndex, day));
    if (
      Number.isNaN(date.getTime()) ||
      date.getUTCFullYear() !== year ||
      date.getUTCMonth() !== monthIndex ||
      date.getUTCDate() !== day
    ) {
      throw new BadRequestException(
        "Dates must be provided in YYYY-MM-DD format"
      );
    }

    return date;
  }

  private resolveMonth(month?: string): {
    start: Date;
    end: Date;
    daysInMonth: number;
    label: string;
    metaMonth: string;
  } {
    let year: number;
    let monthIndex: number;

    if (month) {
      const matches = /^(\d{4})-(\d{2})$/.exec(month);
      if (!matches) {
        throw new BadRequestException(
          "Month must be provided in YYYY-MM format"
        );
      }
      year = Number(matches[1]);
      monthIndex = Number(matches[2]) - 1;
      if (Number.isNaN(year) || Number.isNaN(monthIndex)) {
        throw new BadRequestException(
          "Month must be provided in YYYY-MM format"
        );
      }
    } else {
      const now = new Date();
      year = now.getUTCFullYear();
      monthIndex = now.getUTCMonth();
    }

    if (monthIndex < 0 || monthIndex > 11) {
      throw new BadRequestException("Month value must be between 01 and 12");
    }

    const start = new Date(Date.UTC(year, monthIndex, 1));
    const end = new Date(Date.UTC(year, monthIndex + 1, 1));
    const daysInMonth = new Date(
      Date.UTC(year, monthIndex + 1, 0)
    ).getUTCDate();

    const monthFormatter = new Intl.DateTimeFormat("en-US", {
      month: "long",
    });
    const label = `${monthFormatter.format(start)} ${start.getUTCFullYear()}`;
    const metaMonth = `${year}-${String(monthIndex + 1).padStart(2, "0")}`;

    return { start, end, daysInMonth, label, metaMonth };
  }

  private buildReportColumns(): Array<Record<string, unknown>> {
    const dayColumns = Array.from({ length: 31 }, (_, idx) => {
      const dayNumber = idx + 1;
      return {
        title: `${dayNumber}`,
        dataIndex: ["days", "display", `${dayNumber}`],
        key: `d${dayNumber}`,
        align: "center",
      };
    });

    return [
      { title: "SL", dataIndex: "sl", key: "sl", width: 60, align: "center" },
      { title: "Item", dataIndex: "item", key: "item", width: 180 },
      {
        title: "Image",
        dataIndex: "image",
        key: "image",
        width: 80,
        uiType: "image",
      },
      {
        title: "Size",
        dataIndex: "size",
        key: "size",
        width: 70,
        align: "center",
      },
      {
        title: "Color",
        dataIndex: "color",
        key: "color",
        width: 90,
        align: "center",
      },
      {
        title: "Opening",
        dataIndex: "opening",
        key: "opening",
        width: 90,
        align: "right",
      },
      {
        title: "Days",
        key: "days",
        children: dayColumns,
      },
      {
        title: "Closing",
        dataIndex: "closing",
        key: "closing",
        width: 90,
        align: "right",
      },
    ];
  }
}

function enumerateDateKeys(start: Date, end: Date): string[] {
  const keys: string[] = [];
  for (
    let cursor = new Date(start);
    cursor < end;
    cursor = new Date(cursor.getTime() + 24 * 60 * 60 * 1000)
  ) {
    keys.push(cursor.toISOString().split("T")[0]);
  }
  return keys;
}

function toDateKey(input: Date | string): string {
  const date = new Date(input);
  const normalized = new Date(
    Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate())
  );
  return normalized.toISOString().split("T")[0];
}

function isWithinRange(dateKey: string, start: Date, end: Date): boolean {
  const current = new Date(`${dateKey}T00:00:00.000Z`);
  return (
    current.getTime() >= start.getTime() && current.getTime() < end.getTime()
  );
}
