import {
  ArrayNotEmpty,
  IsArray,
  IsDateString,
  IsEnum,
  IsInt,
  IsNumber,
  IsOptional,
  IsPositive,
  IsString,
  ValidateNested,
} from "class-validator";
import { ApiProperty, ApiPropertyOptional } from "@nestjs/swagger";
import { Type } from "class-transformer";
import type { SupplierProductStatus } from "../../entities/supplier-product.entity";

export class SupplierProductSizeDto {
  @ApiProperty({
    description: "Identifier of the size for this supplier product entry",
    example: 5,
  })
  @IsInt()
  @IsPositive()
  size_id: number;

  @ApiProperty({
    description: "Quantity for the specific size",
    example: 10,
  })
  @IsInt()
  @IsPositive()
  quantity: number;

  @ApiPropertyOptional({
    description:
      "Identifier of the color variant for this size entry. Falls back to the top-level color if omitted.",
    example: 3,
  })
  @IsInt()
  @IsPositive()
  @IsOptional()
  color_id?: number;
}

export class CreateSupplierProductDto {
  @ApiProperty({ description: "The ID of the supplier", example: 1 })
  @IsNumber()
  @IsPositive()
  supplier_id: number;

  @ApiProperty({ description: "The ID of the product", example: 1 })
  @IsNumber()
  @IsPositive()
  product_id: number;

  @ApiProperty({
    description: "The price of the product from this supplier",
    example: 99.99,
  })
  @IsNumber()
  @IsPositive()
  unit_price: number;

  @ApiProperty({
    description: "The quantity of the product from this supplier",
    example: 30,
  })
  @IsNumber()
  @IsPositive()
  @IsOptional()
  quantity?: number;

  @ApiProperty({
    description:
      "Breakdown of sizes with their quantities, optionally specifying color per size.",
    type: [SupplierProductSizeDto],
    example: [
      { size_id: 1, quantity: 10, color_id: 3 },
      { size_id: 2, quantity: 20, color_id: 4 },
    ],
  })
  @IsArray()
  @ArrayNotEmpty()
  @ValidateNested({ each: true })
  @Type(() => SupplierProductSizeDto)
  sizes: SupplierProductSizeDto[];

  @ApiPropertyOptional({
    description: "Current fulfillment status for the supplier product record.",
    enum: ["preorder", "completed"],
    example: "completed",
  })
  @IsOptional()
  @IsEnum(["preorder", "completed"])
  status?: SupplierProductStatus;

  @ApiProperty({
    description: "Additional notes about the supplier product",
    example: "Preferred supplier for bulk orders",
    required: false,
  })
  @IsString()
  @IsOptional()
  note?: string;

  @ApiProperty({
    description: "The date the product was purchased from the supplier",
    example: "2024-03-01",
    required: false,
    type: String,
    format: "date",
  })
  @IsDateString()
  @IsOptional()
  purchase_date?: string;
}
