import {
  Controller,
  Post,
  Body,
  Get,
  Param,
  Delete,
  HttpCode,
  HttpStatus,
  Query,
  ParseIntPipe,
} from "@nestjs/common";
import { SupplierProductService } from "./supplier-product.service";
import { CreateSupplierProductDto } from "./dto/create-supplier-product.dto";
import { CreateBulkSupplierProductDto } from "./dto/create-bulk-supplier-product.dto";
import { SupplierProduct } from "../entities/supplier-product.entity";
import {
  ApiTags,
  ApiOperation,
  ApiResponse,
  ApiParam,
  ApiBody,
  ApiCreatedResponse,
  ApiNotFoundResponse,
  ApiOkResponse,
  ApiNoContentResponse,
  ApiQuery,
  getSchemaPath,
} from "@nestjs/swagger";
import { SupplierProductListQueryDto } from "./dto/list-supplier-product-query.dto";

@ApiTags("Supplier-Products")
@Controller("api/supplier-products")
export class SupplierProductController {
  constructor(
    private readonly supplierProductService: SupplierProductService,
  ) {}

  @Get()
  @ApiOperation({
    summary: "List supplier products with optional search filters",
  })
  @ApiQuery({
    name: "search",
    required: false,
    description: "Search term matched against product name, code, color, or supplier name",
  })
  @ApiQuery({
    name: "supplierId",
    required: false,
    description: "Filter by supplier identifier",
  })
  @ApiOkResponse({
    description: "Paginated supplier product list",
    schema: {
      type: "object",
      properties: {
        data: {
          type: "array",
          items: { $ref: getSchemaPath(SupplierProduct) },
        },
        page: { type: "number" },
        limit: { type: "number" },
        total: { type: "number" },
        previous: { type: "number", nullable: true },
        next: { type: "number", nullable: true },
      },
    },
  })
  listSupplierProducts(
    @Query() query: SupplierProductListQueryDto,
  ) {
    return this.supplierProductService.searchSupplierProducts(query);
  }

  @Get("/product/:productId/options")
  @ApiOperation({
    summary: "Get size and color options available for a specific product",
  })
  @ApiParam({
    name: "productId",
    description: "Identifier of the product",
  })
  @ApiOkResponse({
    description: "Size and color options associated with the product",
    schema: {
      type: "object",
      properties: {
        productId: { type: "number" },
        productName: { type: "string" },
        sizes: {
          type: "array",
          items: {
            type: "object",
            properties: {
              id: { type: "number" },
              name: { type: "string" },
            },
          },
        },
        colors: {
          type: "array",
          items: {
            type: "object",
            properties: {
              id: { type: "number" },
              name: { type: "string" },
            },
          },
        },
      },
    },
  })
  @ApiNotFoundResponse({ description: "Product not found." })
  getProductVariantOptions(
    @Param("productId", ParseIntPipe) productId: number,
  ) {
    return this.supplierProductService.getProductVariantOptions(productId);
  }

  @Post("/assign")
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Assign a product to a supplier" })
  @ApiBody({
    type: CreateSupplierProductDto,
    description: "Data to assign a product to a supplier",
    examples: {
      singleColor: {
        summary: "Single color with per-size quantities",
        value: {
          supplier_id: 1,
          product_id: 1,
          unit_price: 99.99,
          quantity: 30,
          sizes: [
            { size_id: 1, quantity: 10, color_id: 3 },
            { size_id: 2, quantity: 20, color_id: 4 },
          ],
          note: "Preferred supplier for bulk orders",
          purchase_date: "2024-03-01",
        },
      },
      multiColor: {
        summary: "Multiple colors with color-wise sizes",
        value: {
          supplier_id: 1,
          product_id: 2,
          unit_price: 150.0,
          quantity: 25,
          sizes: [
            { size_id: 3, quantity: 12, color_id: 2 },
            { size_id: 4, quantity: 13, color_id: 5 },
          ],
          purchase_date: "2024-05-20",
        },
      },
    },
  })
  @ApiCreatedResponse({ type: SupplierProduct, description: "New supplier inventory record created." })
  @ApiOkResponse({
    type: SupplierProduct,
    description: "Existing supplier inventory restocked; response includes history.",
  })
  @ApiNotFoundResponse({ description: "Supplier or Product not found." })
  @ApiResponse({ status: 400, description: "Bad Request." })
  assignProductToSupplier(
    @Body() createSupplierProductDto: CreateSupplierProductDto,
  ) {
    return this.supplierProductService.assignProductToSupplier(
      createSupplierProductDto,
    );
  }

  

  @Get("/supplier/:supplierId")
  @ApiOperation({ summary: "Get all products assigned to a specific supplier" })
  @ApiParam({ name: "supplierId", description: "ID of the supplier" })
  @ApiOkResponse({ type: [SupplierProduct], description: "Successfully retrieved products for the supplier." })
  @ApiNotFoundResponse({ description: "Supplier not found." })
  findProductsBySupplier(@Param("supplierId") supplierId: string) {
    return this.supplierProductService.findProductsBySupplier(+supplierId);
  }

  @Delete("/supplier/:supplierId/product/:productId")
  @HttpCode(HttpStatus.NO_CONTENT)
  @ApiOperation({ summary: "Remove a product assignment from a supplier" })
  @ApiParam({ name: "supplierId", description: "ID of the supplier" })
  @ApiParam({ name: "productId", description: "ID of the product" })
  @ApiNoContentResponse({ description: "Product assignment successfully removed." })
  @ApiNotFoundResponse({ description: "Assignment not found." })
  removeAssignment(
    @Param("supplierId") supplierId: string,
    @Param("productId") productId: string,
  ) {
    return this.supplierProductService.removeAssignment(
      +supplierId,
      +productId,
    );
  }

  @Post("bulk")
  @ApiOperation({
    summary: "Create multiple supplier products for a single supplier",
  })
  @ApiCreatedResponse({
    description: "Successfully recorded inventory for the supplier.",
    type: [SupplierProduct],
  })
  @ApiBody({
    type: CreateBulkSupplierProductDto,
    description: "Data to create multiple supplier products for a single supplier",
  })
  @ApiNotFoundResponse({ description: "Supplier not found." })
  @ApiResponse({ status: 400, description: "Bad Request." })
  createBulk(
    @Body() createBulkSupplierProductDto: CreateBulkSupplierProductDto,
  ) {
    return this.supplierProductService.createBulk(createBulkSupplierProductDto);
  }
}
