import { SupplierProductService } from "./supplier-product.service";
import { SupplierProduct } from "../entities/supplier-product.entity";
import { Supplier } from "../entities/supplier.entity";
import { Product } from "../entities/product.entity";
import { SupplierProductRestockLog } from "../entities/supplier-product-restock-log.entity";
import { StockService } from "../stock/stock.service";
import { Size } from "../entities/size.entity";
import { SupplierProductSize } from "../entities/supplier-product-size.entity";

describe("SupplierProductService", () => {
  it("increases stock when assigning a supplier product", async () => {
    const supplier = { id: 1 } as Supplier;
    const product = { id: 2 } as Product;
    const sizes = [
      { id: 3, name: "M" } as Size,
      { id: 4, name: "L" } as Size,
    ];

    const supplierRepositoryMock = {
      findOneBy: jest.fn().mockResolvedValue(supplier),
    };
    const productRepositoryMock = {
      findOneBy: jest.fn().mockResolvedValue(product),
    };
    const sizeRepositoryMock = {
      findBy: jest.fn().mockResolvedValue(sizes),
    };
    const colorRepositoryMock = {
      find: jest.fn().mockResolvedValue([]),
      findOne: jest.fn(),
    };

    const supplierProductRepoMock = {
      findOne: jest.fn().mockResolvedValue(null),
      create: jest.fn().mockImplementation((input) => ({
        ...input,
      })),
      save: jest.fn().mockImplementation(async (entity) => ({
        ...entity,
        id: 10,
      })),
      findOneOrFail: jest.fn().mockResolvedValue({
        id: 10,
        product,
        supplier,
        sizes: [
          { size: sizes[0], quantity: 5 },
          { size: sizes[1], quantity: 7 },
        ],
      }),
    };

    const restockLogRepoMock = {
      create: jest.fn().mockReturnValue({ id: 20 }),
      save: jest.fn().mockResolvedValue(undefined),
    };

    const supplierProductSizeRepoMock = {
      create: jest.fn().mockImplementation((input) => input),
      save: jest.fn().mockImplementation(async (entity) => entity),
    };

    const managerMock = {
      getRepository: jest.fn().mockImplementation((entity) => {
        if (entity === SupplierProduct) {
          return supplierProductRepoMock;
        }
        if (entity === SupplierProductRestockLog) {
          return restockLogRepoMock;
        }
        if (entity === SupplierProductSize) {
          return supplierProductSizeRepoMock;
        }
        throw new Error("Unexpected repository request");
      }),
    };

    const transactionMock = jest
      .fn()
      .mockImplementation(async (callback) => callback(managerMock));

    const supplierProductRepositoryMock = {
      manager: { transaction: transactionMock },
    };

    const stockServiceMock = {
      increaseStock: jest.fn().mockResolvedValue(undefined),
    };

    const service = new SupplierProductService(
      supplierProductRepositoryMock as any,
      supplierRepositoryMock as any,
      productRepositoryMock as any,
      sizeRepositoryMock as any,
      colorRepositoryMock as any,
      stockServiceMock as unknown as StockService
    );

    await service.assignProductToSupplier({
      supplier_id: 1,
      product_id: 2,
      quantity: 12,
      unit_price: 100,
      sizes: [
        { size_id: 3, quantity: 5 },
        { size_id: 4, quantity: 7 },
      ],
    });

    expect(stockServiceMock.increaseStock).toHaveBeenNthCalledWith(
      1,
      expect.objectContaining({
        productId: 2,
        sizeId: 3,
        quantity: 5,
        colorId: null,
        movementType: "NEW",
        referenceType: "SUPPLIER_PRODUCT",
        referenceId: 10,
        createIfMissing: true,
        manager: managerMock,
      })
    );

    expect(stockServiceMock.increaseStock).toHaveBeenNthCalledWith(
      2,
      expect.objectContaining({
        productId: 2,
        sizeId: 4,
        quantity: 7,
        colorId: null,
        movementType: "NEW",
        referenceType: "SUPPLIER_PRODUCT",
        referenceId: 10,
        createIfMissing: true,
        manager: managerMock,
      })
    );

    expect(sizeRepositoryMock.findBy).toHaveBeenCalledTimes(1);
  });
});
