import {
  BadRequestException,
  Injectable,
  NotFoundException,
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { FindOptionsWhere, In, IsNull, Repository } from "typeorm";
import {
  SupplierProduct,
  SupplierProductStatus,
} from "../entities/supplier-product.entity";
import { CreateSupplierProductDto } from "./dto/create-supplier-product.dto";
import { CreateBulkSupplierProductDto } from "./dto/create-bulk-supplier-product.dto";
import { Supplier } from "../entities/supplier.entity";
import { Product } from "../entities/product.entity";
import { SupplierProductRestockLog } from "../entities/supplier-product-restock-log.entity";
import { StockService } from "../stock/stock.service";
import { Size } from "../entities/size.entity";
import { SupplierProductSize } from "../entities/supplier-product-size.entity";
import { Color } from "../entities/color.entity";
import { SupplierProductListQueryDto } from "./dto/list-supplier-product-query.dto";

@Injectable()
export class SupplierProductService {
  constructor(
    @InjectRepository(SupplierProduct)
    private readonly supplierProductRepository: Repository<SupplierProduct>,
    @InjectRepository(Supplier)
    private readonly supplierRepository: Repository<Supplier>,
    @InjectRepository(Product)
    private readonly productRepository: Repository<Product>,
    @InjectRepository(Size)
    private readonly sizeRepository: Repository<Size>,
    @InjectRepository(Color)
    private readonly colorRepository: Repository<Color>,
    private readonly stockService: StockService
  ) {}

  async assignProductToSupplier(
    createSupplierProductDto: CreateSupplierProductDto
  ): Promise<SupplierProduct> {
    const { supplier_id, product_id, sizes } = createSupplierProductDto;

    const supplier = await this.supplierRepository.findOneBy({
      id: supplier_id,
    });
    if (!supplier) {
      throw new NotFoundException(
        `Supplier with ID ${supplier_id} not found`
      );
    }

    const product = await this.productRepository.findOneBy({ id: product_id });
    if (!product) {
      throw new NotFoundException(
        `Product with ID ${product_id} not found`
      );
    }

    const sizeRequests = this.extractSizeRequests(createSupplierProductDto);
    const resolvedSizes = await this.resolveSizes(
      sizeRequests,
      null
    );

    return this.createOrRestockSupplierProduct(supplier, product, {
      unit_price: createSupplierProductDto.unit_price,
      sizes: resolvedSizes,
      color: null,
      hasExplicitColor: false,
      status: createSupplierProductDto.status,
      note: createSupplierProductDto.note,
      purchase_date: createSupplierProductDto.purchase_date,
      requestedTotalQuantity: createSupplierProductDto.quantity,
    });
  }

  private async createOrRestockSupplierProduct(
    supplier: Supplier,
    product: Product,
    payload: {
      unit_price: number;
      sizes: Array<{ size: Size; quantity: number; color: Color | null }>;
      color?: Color | null;
      hasExplicitColor: boolean;
      status?: SupplierProductStatus;
      note?: string;
      purchase_date?: string;
      requestedTotalQuantity?: number;
    }
  ): Promise<SupplierProduct> {
    return this.supplierProductRepository.manager.transaction(
      async (manager) => {
        const totalQuantity = payload.sizes.reduce(
          (sum, entry) => sum + entry.quantity,
          0
        );
        const derivedColor = this.derivePrimaryColor(payload.sizes);
        const recordColor = payload.hasExplicitColor
          ? payload.color ?? null
          : derivedColor;

        if (
          payload.requestedTotalQuantity !== undefined &&
          payload.requestedTotalQuantity !== totalQuantity
        ) {
          throw new BadRequestException(
            `Provided quantity (${payload.requestedTotalQuantity}) does not match total of sizes (${totalQuantity}).`
          );
        }

        const supplierProductRepo = manager.getRepository(SupplierProduct);
        const restockRepo = manager.getRepository(
          SupplierProductRestockLog
        );
        const supplierProductSizeRepo =
          manager.getRepository(SupplierProductSize);

        const whereClause: FindOptionsWhere<SupplierProduct> = {
          supplier: { id: supplier.id },
          product: { id: product.id },
          color_id: recordColor ? recordColor.id : undefined,
        };
        if (!recordColor) {
          whereClause.color_id = IsNull();
        }

        let supplierProduct = await supplierProductRepo.findOne({
          where: whereClause,
          relations: {
            sizes: { size: true, color: true },
            color: true,
          },
        });

        if (!supplierProduct) {
          supplierProduct = supplierProductRepo.create({
            supplier,
            product,
            unit_price: payload.unit_price,
            quantity: totalQuantity,
            color: recordColor ?? null,
            color_id: recordColor?.id ?? null,
            status: payload.status ?? "completed",
            note: payload.note ?? null,
            purchase_date: payload.purchase_date
              ? new Date(payload.purchase_date)
              : null,
          });
          supplierProduct.sizes = [];
        } else {
          supplierProduct.quantity += totalQuantity;
          supplierProduct.unit_price = payload.unit_price;
          supplierProduct.color = recordColor ?? null;
          supplierProduct.color_id = recordColor?.id ?? null;
          if (payload.status) {
            supplierProduct.status = payload.status;
          }
          if (payload.note !== undefined) {
            supplierProduct.note = payload.note;
          }
          if (payload.purchase_date !== undefined) {
            supplierProduct.purchase_date = payload.purchase_date
              ? new Date(payload.purchase_date)
              : null;
          }
        }

        const savedSupplierProduct = await supplierProductRepo.save(
          supplierProduct
        );

        const existingSizeMap = new Map<string, SupplierProductSize>();
        if (supplierProduct.sizes) {
          for (const sizeEntry of supplierProduct.sizes) {
            const key = this.buildVariantKey(
              sizeEntry.size_id,
              sizeEntry.color_id ?? null
            );
            existingSizeMap.set(key, sizeEntry);
          }
        }

        for (const entry of payload.sizes) {
          const variantKey = this.buildVariantKey(
            entry.size.id,
            entry.color?.id ?? null
          );
          const existingSize = existingSizeMap.get(variantKey);
          if (existingSize) {
            existingSize.quantity += entry.quantity;
            await supplierProductSizeRepo.save(existingSize);
          } else {
            const newSize = supplierProductSizeRepo.create({
              supplierProduct: savedSupplierProduct,
              supplier_product_id: savedSupplierProduct.id,
              size: entry.size,
              size_id: entry.size.id,
              quantity: entry.quantity,
              color: entry.color ?? null,
              color_id: entry.color?.id ?? null,
            });
            const savedSize = await supplierProductSizeRepo.save(newSize);
            existingSizeMap.set(variantKey, savedSize);
          }

          await this.stockService.increaseStock({
            productId: product.id,
            sizeId: entry.size.id,
            quantity: entry.quantity,
            colorId: entry.color?.id ?? recordColor?.id ?? null,
            movementType: "NEW",
            manager,
            referenceType: "SUPPLIER_PRODUCT",
            referenceId: savedSupplierProduct.id,
            createIfMissing: true,
          });
        }

        const restockLog = restockRepo.create({
          supplierProduct: savedSupplierProduct,
          quantity: totalQuantity,
          unit_price: payload.unit_price,
        });
        await restockRepo.save(restockLog);

        return supplierProductRepo.findOneOrFail({
          where: { id: savedSupplierProduct.id },
          relations: {
            product: true,
            supplier: true,
            color: true,
            sizes: { size: true, color: true },
            restockLogs: true,
          },
          order: {
            restockLogs: { restocked_at: "DESC" },
          },
        });
      }
    );
  }

  private async resolveSizes(
    sizes: Array<{ size_id: number; quantity: number; color_id?: number | null }>,
    defaultColorId: number | null
  ): Promise<
    Array<{ size: Size; quantity: number; color: Color | null }>
  > {
    const uniqueIds = Array.from(new Set(sizes.map((item) => item.size_id)));
    const sizeEntities = await this.sizeRepository.findBy({
      id: In(uniqueIds),
    });

    if (sizeEntities.length !== uniqueIds.length) {
      const foundIds = new Set(sizeEntities.map((size) => size.id));
      const missingId = uniqueIds.find((id) => !foundIds.has(id));
      throw new NotFoundException(
        `Size with ID ${missingId ?? "unknown"} not found`
      );
    }

    const sizeMap = new Map(sizeEntities.map((size) => [size.id, size]));
    const colorIds = new Set<number>();
    if (defaultColorId !== null) {
      colorIds.add(defaultColorId);
    }
    for (const sizeDto of sizes) {
      if (sizeDto.color_id !== undefined && sizeDto.color_id !== null) {
        colorIds.add(sizeDto.color_id);
      }
    }

    const colorEntities = colorIds.size
      ? await this.colorRepository.find({
          where: { id: In(Array.from(colorIds)) },
        })
      : [];

    if (colorIds.size && colorEntities.length !== colorIds.size) {
      const resolvedIds = new Set(colorEntities.map((color) => color.id));
      const missing = Array.from(colorIds).filter((id) => !resolvedIds.has(id));
      throw new NotFoundException(
        `Colors not found for identifiers: ${missing.join(", ")}`
      );
    }

    const colorMap = new Map(colorEntities.map((color) => [color.id, color]));
    const defaultColor =
      defaultColorId !== null ? colorMap.get(defaultColorId) ?? null : null;

    const aggregated = new Map<
      string,
      { size: Size; quantity: number; color: Color | null }
    >();

    for (const sizeDto of sizes) {
      const size = sizeMap.get(sizeDto.size_id);
      if (!size) {
        throw new NotFoundException(
          `Size with ID ${sizeDto.size_id} not found`
        );
      }

      let resolvedColor: Color | null = defaultColor ?? null;
      if (sizeDto.color_id !== undefined) {
        if (sizeDto.color_id === null) {
          resolvedColor = null;
        } else {
          const color = colorMap.get(sizeDto.color_id);
          if (!color) {
            throw new NotFoundException(
              `Color with ID ${sizeDto.color_id} not found`
            );
          }
          resolvedColor = color;
        }
      }

      const key = this.buildVariantKey(
        size.id,
        resolvedColor?.id ?? null
      );
      const existing = aggregated.get(key);
      if (existing) {
        existing.quantity += sizeDto.quantity;
      } else {
        aggregated.set(key, {
          size,
          quantity: sizeDto.quantity,
          color: resolvedColor,
        });
      }
    }

    return Array.from(aggregated.values());
  }

  async searchSupplierProducts(
    query: SupplierProductListQueryDto
  ): Promise<{
    data: SupplierProduct[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(Number.parseInt(query.page ?? "1", 10) || 1, 1);
    const limit = Math.max(Number.parseInt(query.limit ?? "10", 10) || 10, 1);
    const search = query.searchTerm.toLowerCase();

    const supplierId = query.supplierId
      ? Number.parseInt(query.supplierId, 10)
      : undefined;

    if (supplierId !== undefined && (Number.isNaN(supplierId) || supplierId <= 0)) {
      throw new BadRequestException("supplierId must be a positive number when provided");
    }

    const queryBuilder = this.supplierProductRepository
      .createQueryBuilder("supplierProduct")
      .leftJoinAndSelect("supplierProduct.product", "product")
      .leftJoinAndSelect("supplierProduct.supplier", "supplier")
      .leftJoinAndSelect("supplierProduct.color", "color")
      .leftJoinAndSelect("supplierProduct.sizes", "variant")
      .leftJoinAndSelect("variant.size", "size")
      .leftJoinAndSelect("variant.color", "variantColor")
      .leftJoinAndSelect("supplierProduct.restockLogs", "restockLogs")
      .orderBy("supplierProduct.created_at", "DESC")
      .addOrderBy("supplierProduct.id", "DESC")
      .distinct(true);

    if (supplierId !== undefined) {
      queryBuilder.andWhere("supplierProduct.supplier_id = :supplierId", {
        supplierId,
      });
    }

    if (search) {
      queryBuilder.andWhere(
        "(LOWER(product.product_name) LIKE :search OR LOWER(product.product_code) LIKE :search OR LOWER(product.product_color) LIKE :search OR LOWER(supplier.name) LIKE :search)",
        { search: `%${search}%` }
      );
    }

    const [data, total] = await queryBuilder
      .skip((page - 1) * limit)
      .take(limit)
      .getManyAndCount();

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async getProductVariantOptions(
    productId: number
  ): Promise<{
    productId: number;
    productName: string;
    sizes: Array<{ id: number; name: string }>;
    colors: Array<{ id: number; name: string }>;
  }> {
    const product = await this.productRepository.findOne({
      where: { id: productId },
      relations: { sizes: true, colors: true },
    });

    if (!product) {
      throw new NotFoundException(`Product with ID ${productId} not found`);
    }

    const sizeMap = new Map<number, Size>();
    for (const size of product.sizes ?? []) {
      sizeMap.set(size.id, size);
    }

    const colorMap = new Map<number, Color>();
    for (const color of product.colors ?? []) {
      colorMap.set(color.id, color);
    }

    const supplierVariants = await this.supplierProductRepository.find({
      where: { product: { id: productId } },
      relations: {
        sizes: { size: true, color: true },
        color: true,
      },
    });

    for (const variant of supplierVariants) {
      if (variant.color) {
        colorMap.set(variant.color.id, variant.color);
      }
      for (const sizeEntry of variant.sizes ?? []) {
        if (sizeEntry.size) {
          sizeMap.set(sizeEntry.size.id, sizeEntry.size);
        }
        if (sizeEntry.color) {
          colorMap.set(sizeEntry.color.id, sizeEntry.color);
        }
      }
    }

    const sizes = Array.from(sizeMap.values()).sort((a, b) =>
      a.name.localeCompare(b.name)
    );
    const colors = Array.from(colorMap.values()).sort((a, b) =>
      a.name.localeCompare(b.name)
    );

    return {
      productId: product.id,
      productName: product.product_name,
      sizes: sizes.map((size) => ({ id: size.id, name: size.name })),
      colors: colors.map((color) => ({ id: color.id, name: color.name })),
    };
  }

  async findProductsBySupplier(supplierId: number): Promise<SupplierProduct[]> {
    const supplier = await this.supplierRepository.findOneBy({
      id: supplierId,
    });
    if (!supplier) {
      throw new NotFoundException(`Supplier with ID ${supplierId} not found`);
    }
    return this.supplierProductRepository.find({
      where: { supplier: { id: supplierId } },
      relations: {
        product: true,
        supplier: true,
        color: true,
        sizes: { size: true, color: true },
        restockLogs: true,
      },
      order: {
        created_at: "DESC",
        restockLogs: { restocked_at: "DESC" },
      },
    });
  }

  async removeAssignment(supplierId: number, productId: number): Promise<void> {
    const result = await this.supplierProductRepository.delete({
      supplier: { id: supplierId },
      product: { id: productId },
    });

    if (result.affected === 0) {
      throw new NotFoundException(
        `Assignment of product ${productId} to supplier ${supplierId} not found`
      );
    }
  }

  async createBulk(
    createBulkSupplierProductDto: CreateBulkSupplierProductDto
  ): Promise<SupplierProduct[]> {
    const { supplier_id, products } = createBulkSupplierProductDto;

    const supplier = await this.supplierRepository.findOneBy({
      id: supplier_id,
    });
    if (!supplier) {
      throw new NotFoundException(
        `Supplier with ID ${supplier_id} not found`
      );
    }

    const supplierProductsToCreate: SupplierProduct[] = [];

    for (const productDto of products) {
      const product = await this.productRepository.findOneBy({
        id: productDto.product_id,
      });
      if (!product) {
        throw new NotFoundException(
          `Product with ID ${productDto.product_id} not found`
        );
      }

      const sizeRequests = this.extractSizeRequests(productDto);

      const resolvedSizes = await this.resolveSizes(sizeRequests, null);

      const existingSupplierProduct =
        await this.createOrRestockSupplierProduct(supplier, product, {
          unit_price: productDto.unit_price,
          sizes: resolvedSizes,
          color: null,
          hasExplicitColor: false,
          status: productDto.status,
          note: productDto.note,
          purchase_date: productDto.purchase_date,
          requestedTotalQuantity: productDto.quantity,
        });
      supplierProductsToCreate.push(existingSupplierProduct);
    }

    return supplierProductsToCreate;
  }

  private extractSizeRequests(
    dto: Pick<CreateSupplierProductDto, "sizes">
  ): Array<{ size_id: number; quantity: number; color_id?: number | null }> {
    const entries: Array<{
      size_id: number;
      quantity: number;
      color_id?: number | null;
    }> = [];

    if (Array.isArray(dto.sizes)) {
      for (const item of dto.sizes) {
        entries.push({
          size_id: item.size_id,
          quantity: item.quantity,
          color_id: item.color_id,
        });
      }
    }

    if (entries.length === 0) {
      throw new BadRequestException(
        "At least one size entry is required."
      );
    }

    return entries;
  }

  private buildVariantKey(sizeId: number, colorId: number | null): string {
    return `${sizeId}::${colorId ?? "null"}`;
  }

  private derivePrimaryColor(
    sizes: Array<{ color: Color | null }>
  ): Color | null {
    const colorIds = new Set<number>();
    let firstColor: Color | null = null;
    let hasNull = false;

    for (const entry of sizes) {
      if (!entry.color) {
        hasNull = true;
        continue;
      }
      colorIds.add(entry.color.id);
      if (!firstColor) {
        firstColor = entry.color;
      } else if (entry.color.id !== firstColor.id) {
        return null;
      }
    }

    if (hasNull) {
      return null;
    }

    if (colorIds.size === 1 && firstColor) {
      return firstColor;
    }

    return null;
  }

}
