import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Patch,
  Post,
  Query,
} from "@nestjs/common";
import { ApiOperation, ApiQuery, ApiResponse, ApiTags } from "@nestjs/swagger";
import { PaginationDto } from "../common/dto/pagination.dto";
import { CreateSupplierDto } from "./dto/create-supplier.dto";
import { UpdateSupplierDto } from "./dto/update-supplier.dto";
import { SupplierService } from "./supplier.service";

@ApiTags("Suppliers")
@Controller("api/suppliers")
export class SupplierController {
  constructor(private readonly supplierService: SupplierService) {}

  @Post()
  @HttpCode(HttpStatus.CREATED)
  @ApiOperation({ summary: "Create a new supplier" })
  @ApiResponse({
    status: 201,
    description: "The supplier has been successfully created.",
  })
  @ApiResponse({ status: 400, description: "Bad Request." })
  create(@Body() createSupplierDto: CreateSupplierDto) {
    return this.supplierService.create(createSupplierDto);
  }

  @Get()
  @ApiOperation({
    summary: "Retrieve all suppliers with pagination and search",
  })
  @ApiResponse({
    status: 200,
    description: "Returns all suppliers with pagination and search.",
  })
  @ApiQuery({
    name: "limit",
    type: Number,
    required: false,
    description: "Limit of items per page",
  })
  @ApiQuery({
    name: "page",
    type: Number,
    required: false,
    description: "Page of items",
  })
  @ApiQuery({
    name: "query",
    type: String,
    required: false,
    description: "Search keyword (alias: search)",
  })
  @ApiQuery({
    name: "search",
    type: String,
    required: false,
    description: "Deprecated search keyword; prefer query",
  })
  findAll(@Query() paginationDto: PaginationDto) {
    return this.supplierService.findAll(paginationDto);
  }

  @Get(":id")
  @ApiOperation({ summary: "Retrieve a supplier by ID" })
  @ApiResponse({
    status: 200,
    description: "Returns the supplier with the specified ID.",
  })
  @ApiResponse({ status: 404, description: "Supplier not found." })
  findOne(@Param("id") id: string) {
    return this.supplierService.findOne(+id);
  }

  @Patch(":id")
  @ApiOperation({ summary: "Update a supplier by ID" })
  @ApiResponse({
    status: 200,
    description: "The supplier has been successfully updated.",
  })
  @ApiResponse({ status: 404, description: "Supplier not found." })
  @ApiResponse({ status: 400, description: "Bad Request." })
  update(
    @Param("id") id: string,
    @Body() updateSupplierDto: UpdateSupplierDto,
  ) {
    return this.supplierService.update(+id, updateSupplierDto);
  }

  @Delete(":id")
  async remove(@Param("id") id: string) {
    await this.supplierService.remove(+id);
    return { success: true, message: "Supplier deleted successfully" };
  }
}
