import {
  ConflictException,
  Injectable,
  NotFoundException,
} from "@nestjs/common";
import { InjectRepository } from "@nestjs/typeorm";
import { Repository, DeepPartial } from "typeorm";
import { Supplier } from "../entities/supplier.entity";
import { CreateSupplierDto } from "./dto/create-supplier.dto";
import { UpdateSupplierDto } from "./dto/update-supplier.dto";
import { PaginationDto } from "../common/dto/pagination.dto";

@Injectable()
export class SupplierService {
  constructor(
    @InjectRepository(Supplier)
    private supplierRepository: Repository<Supplier>,
  ) {}

  async create(createSupplierDto: CreateSupplierDto): Promise<Supplier> {
    const { email, phone } = createSupplierDto;

    const existingSupplier = await this.supplierRepository.findOne({
      where: [{ email }, { phone }],
    });

    if (existingSupplier) {
      if (existingSupplier.email === email) {
        throw new ConflictException(
          `Supplier with email "${email}" already exists.`,
        );
      }
      if (existingSupplier.phone === phone) {
        throw new ConflictException(
          `Supplier with phone "${phone}" already exists.`,
        );
      }
    }

    const supplier = new Supplier();
    Object.assign(supplier, createSupplierDto);
    return this.supplierRepository.save(supplier);
  }

  async findAll(
    paginationDto: PaginationDto,
  ): Promise<{
    data: Supplier[];
    page: number;
    limit: number;
    total: number;
    previous: number | null;
    next: number | null;
  }> {
    const page = Math.max(
      Number.parseInt(paginationDto.page ?? "1", 10) || 1,
      1,
    );
    const limit = Math.max(
      Number.parseInt(paginationDto.limit ?? "10", 10) || 10,
      1,
    );
    const search = paginationDto.searchTerm;

    const queryBuilder = this.supplierRepository
      .createQueryBuilder("supplier")
      .orderBy("supplier.created_at", "DESC");

    if (search) {
      queryBuilder.where(
        "(LOWER(supplier.name) LIKE LOWER(:search) OR LOWER(supplier.email) LIKE LOWER(:search) OR LOWER(supplier.phone) LIKE LOWER(:search))",
        { search: `%${search}%` },
      );
    }

    const [data, total] = await queryBuilder
      .skip((page - 1) * limit)
      .take(limit)
      .getManyAndCount();

    const totalPages = total > 0 ? Math.ceil(total / limit) : 0;

    return {
      data,
      page,
      limit,
      total,
      previous: page > 1 ? page - 1 : null,
      next: page < totalPages ? page + 1 : null,
    };
  }

  async findOne(id: number) {
    const supplier = await this.supplierRepository.findOneBy({ id });
    if (!supplier) {
      throw new NotFoundException(`Supplier with id: ${id} not found`);
    }
    return supplier;
  }

  async update(
    id: number,
    updateSupplierDto: UpdateSupplierDto,
  ): Promise<Supplier> {
    const supplier = await this.findOne(id);
    this.supplierRepository.merge(
      supplier,
      updateSupplierDto as DeepPartial<Supplier>,
    );
    return this.supplierRepository.save(supplier);
  }

  async remove(id: number): Promise<void> {
    const result = await this.supplierRepository.delete(id);
    if (result.affected === 0) {
      throw new NotFoundException(`Supplier with ID ${id} not found`);
    }
  }
}
