import { Test, TestingModule } from "@nestjs/testing";
import { INestApplication } from "@nestjs/common";
import request from "supertest";
import { AppModule } from "./../src/app.module";
import { DataSource } from "typeorm";
import { Supplier } from "../src/entities/supplier.entity";
import { Product } from "../src/entities/product.entity";
import { PurchaseOrder } from "../src/entities/purchase-order.entity";
import { PurchaseOrderItem } from "../src/entities/purchase-order-item.entity";

describe("PurchaseOrderController (e2e)", () => {
  let app: INestApplication;
  let dataSource: DataSource;
  let supplier: Supplier;
  let product1: Product;
  let product2: Product;

  beforeAll(async () => {
    const moduleFixture: TestingModule = await Test.createTestingModule({
      imports: [AppModule],
    }).compile();

    app = moduleFixture.createNestApplication();
    await app.init();

    dataSource = app.get(DataSource);

    // Clear database before tests
    await dataSource.synchronize(true); // Use synchronize to clear and recreate schema

    // Seed data
    const supplierRepository = dataSource.getRepository(Supplier);
    const productRepository = dataSource.getRepository(Product);

    supplier = await supplierRepository.save({
      name: "Test Supplier",
      email: "test@supplier.com",
    });

    product1 = await productRepository.save({
      product_code: "TP1",
      product_name: "Test Product 1",
    });

    product2 = await productRepository.save({
      product_code: "TP2",
      product_name: "Test Product 2",
    });
  });

  afterAll(async () => {
    await dataSource.destroy();
    await app.close();
  });

  beforeEach(async () => {
    // Clear purchase orders and items before each test
    await dataSource.query('SET FOREIGN_KEY_CHECKS = 0');
    await dataSource.getRepository(PurchaseOrderItem).clear();
    await dataSource.getRepository(PurchaseOrder).clear();
    await dataSource.query('SET FOREIGN_KEY_CHECKS = 1');
  });

  it("/api/orders (POST) - should create a purchase order and calculate total amount", async () => {
    const createOrderDto = {
      supplier_id: supplier.id,
      order_date: new Date().toISOString().slice(0, 10),
      items: [
        {
          product_id: product1.id,
          size: "M",
          quantity: 2,
          unit_price: 10.0,
        },
        {
          product_id: product2.id,
          size: "L",
          quantity: 3,
          unit_price: 20.0,
        },
      ],
    };

    const expectedTotalAmount = 2 * 10.0 + 3 * 20.0; // 20 + 60 = 80

    const response = await request(app.getHttpServer())
      .post("/api/orders")
      .send(createOrderDto)
      .expect(201);

    expect(response.body).toBeDefined();
    expect(response.body.total_amount).toBe(expectedTotalAmount.toString() + ".00");
    expect(response.body.purchaseOrderItems).toHaveLength(2);
    expect(response.body.purchaseOrderItems[0].total).toBe((2 * 10.0).toString() + ".00");
    expect(response.body.purchaseOrderItems[1].total).toBe((3 * 20.0).toString() + ".00");
  });

  it("/api/orders/:id (GET) - should return a purchase order with items", async () => {
    const createOrderDto = {
      supplier_id: supplier.id,
      order_date: new Date().toISOString().slice(0, 10),
      items: [
        {
          product_id: product1.id,
          size: "S",
          quantity: 1,
          unit_price: 10.0,
        },
      ],
    };

    const createResponse = await request(app.getHttpServer())
      .post("/api/orders")
      .send(createOrderDto)
      .expect(201);

    const orderId = createResponse.body.id;

    const getResponse = await request(app.getHttpServer())
      .get(`/api/orders/${orderId}`)
      .expect(200);

    expect(getResponse.body).toBeDefined();
    expect(getResponse.body.id).toBe(orderId);
    expect(getResponse.body.purchaseOrderItems).toHaveLength(1);
    expect(getResponse.body.purchaseOrderItems[0].product.id).toBe(product1.id);
  });
});
